"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isEffectedPolicySelected = exports.forceHTMLElementOffsetWidth = exports.clickOnEffectedPolicy = void 0;
var _userEvent = _interopRequireDefault(require("@testing-library/user-event"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Forces the `offsetWidth` of `HTMLElement` to a given value. Needed due to the use of
 * `react-virtualized-auto-sizer` by the eui `Selectable` component
 *
 * @param [width=100]
 * @returns reset(): void
 *
 * @example
 * const resetEnv = forceHTMLElementOffsetWidth();
 * //... later
 * resetEnv();
 */
const forceHTMLElementOffsetWidth = (width = 100) => {
  const currentOffsetDefinition = Object.getOwnPropertyDescriptor(window.HTMLElement.prototype, 'offsetWidth');
  Object.defineProperties(window.HTMLElement.prototype, {
    offsetWidth: {
      ...(currentOffsetDefinition || {}),
      get() {
        return width;
      }
    }
  });
  return function reset() {
    if (currentOffsetDefinition) {
      Object.defineProperties(window.HTMLElement.prototype, {
        offsetWidth: {
          ...(currentOffsetDefinition || {})
        }
      });
    }
  };
};

/**
 * Clicks on a policy being displayed when `per policy` is selected.
 * NOTE: ensure that per-policy is selected first. This utility will
 * not do that.
 * @param renderResult
 * @param [atIndex=0]
 */
exports.forceHTMLElementOffsetWidth = forceHTMLElementOffsetWidth;
const clickOnEffectedPolicy = async (renderResult, atIndex = 0) => {
  const policiesHolderElement = await renderResult.findByTestId('effectedPolicies-select-policiesSelectable');
  const policyElements = policiesHolderElement.querySelectorAll('li.euiSelectableListItem');
  const item = policyElements.item(atIndex);
  if (item) {
    _userEvent.default.click(item);
  }
  return item;
};

/**
 * Returns true or false as to whether an effect policy at a given index in the list is selected or not
 * @param renderResult
 * @param atIndex
 */
exports.clickOnEffectedPolicy = clickOnEffectedPolicy;
const isEffectedPolicySelected = async (renderResult, atIndex = 0) => {
  const policiesHolderElement = await renderResult.findByTestId('effectedPolicies-select-policiesSelectable');
  const policyElements = policiesHolderElement.querySelectorAll('li.euiSelectableListItem');
  const item = policyElements.item(atIndex);
  if (!item) {
    throw new Error(`No policy found in EffectedPolicySelect at index position ${atIndex}`);
  }
  return item.dataset.testSelected === 'true';
};
exports.isEffectedPolicySelected = isEffectedPolicySelected;