"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useArtifactCardPropsProvider = void 0;
var _react = require("react");
var _artifact_entry_card = require("../../artifact_entry_card");
var _use_test_id_generator = require("../../../hooks/use_test_id_generator");
var _hooks = require("../../../services/policies/hooks");
var _translations = require("../../../common/translations");
var _kibana = require("../../../../common/lib/kibana");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Return a function that can be used to retrieve props for an `ArtifactCardEntry` component given an
 * `ExceptionListItemSchema` on input
 */
const useArtifactCardPropsProvider = ({
  items,
  onAction,
  cardActionDeleteLabel,
  cardActionEditLabel,
  dataTestSubj,
  allowCardDeleteAction = true,
  allowCardEditAction = true
}) => {
  const getTestId = (0, _use_test_id_generator.useTestIdGenerator)(dataTestSubj);
  const toasts = (0, _kibana.useToasts)();
  const {
    data: policyData
  } = (0, _hooks.useGetEndpointSpecificPolicies)({
    onError: error => {
      toasts.addDanger((0, _translations.getLoadPoliciesError)(error));
    }
  });
  const policies = (0, _artifact_entry_card.useEndpointPoliciesToArtifactPolicies)(policyData === null || policyData === void 0 ? void 0 : policyData.items);
  const artifactCardPropsPerItem = (0, _react.useMemo)(() => {
    const cachedCardProps = {};

    // Casting `listItems` below to remove the `Immutable<>` from it in order to prevent errors
    // with common component's props
    for (const artifactItem of items) {
      const cardActions = [];
      if (allowCardEditAction) {
        cardActions.push({
          icon: 'controlsHorizontal',
          onClick: () => {
            onAction({
              type: 'edit',
              item: artifactItem
            });
          },
          'data-test-subj': getTestId('cardEditAction'),
          children: cardActionEditLabel
        });
      }
      if (allowCardDeleteAction) {
        cardActions.push({
          icon: 'trash',
          onClick: () => {
            onAction({
              type: 'delete',
              item: artifactItem
            });
          },
          'data-test-subj': getTestId('cardDeleteAction'),
          children: cardActionDeleteLabel
        });
      }
      cachedCardProps[artifactItem.id] = {
        item: artifactItem,
        policies,
        'data-test-subj': dataTestSubj,
        actions: cardActions.length > 0 ? cardActions : undefined,
        hideDescription: !artifactItem.description,
        hideComments: !artifactItem.comments.length
      };
    }
    return cachedCardProps;
  }, [items, allowCardEditAction, allowCardDeleteAction, policies, dataTestSubj, getTestId, cardActionEditLabel, onAction, cardActionDeleteLabel]);
  return (0, _react.useCallback)(artifactItem => {
    return artifactCardPropsPerItem[artifactItem.id];
  }, [artifactCardPropsPerItem]);
};
exports.useArtifactCardPropsProvider = useArtifactCardPropsProvider;