"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateSingleAction = exports.validateRuleActionsField = exports.debouncedValidateRuleActionsField = exports.DEFAULT_VALIDATION_TIMEOUT = void 0;
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/* istanbul ignore file */

const DEFAULT_VALIDATION_TIMEOUT = 100;
exports.DEFAULT_VALIDATION_TIMEOUT = DEFAULT_VALIDATION_TIMEOUT;
const validateSingleAction = async (actionItem, actionTypeRegistry) => {
  const actionParamsErrors = await (0, _utils.validateActionParams)(actionItem, actionTypeRegistry);
  const mustacheErrors = (0, _utils.validateMustache)(actionItem.params);
  return [...actionParamsErrors, ...mustacheErrors];
};
exports.validateSingleAction = validateSingleAction;
const validateRuleActionsField = actionTypeRegistry => async (...data) => {
  const [{
    value,
    path
  }] = data;
  const errors = [];
  for (const actionItem of value) {
    const errorsArray = await validateSingleAction(actionItem, actionTypeRegistry);
    if (errorsArray.length) {
      const actionTypeName = (0, _utils.getActionTypeName)(actionItem.actionTypeId);
      const errorsListItems = errorsArray.map(error => `*   ${error}\n`);
      errors.push(`\n**${actionTypeName}:**\n${errorsListItems.join('')}`);
    }
  }
  if (errors.length) {
    return {
      code: 'ERR_FIELD_FORMAT',
      path,
      message: `${errors.join('\n')}`
    };
  }
};

/**
 * Debounces validation by canceling previous validation requests. Essentially leveraging the async validation
 * cancellation behavior from the hook_form_lib. Necessary to prevent error validation flashing when first adding an
 * action until root cause of https://github.com/elastic/kibana/issues/142217 is found
 *
 * See docs for details:
 * https://docs.elastic.dev/form-lib/examples/validation#cancel-asynchronous-validation
 *
 * Note: _.throttle/debounce does not have async support, and so not used https://github.com/lodash/lodash/issues/4815.
 *
 * @param actionTypeRegistry
 * @param defaultValidationTimeout
 */
exports.validateRuleActionsField = validateRuleActionsField;
const debouncedValidateRuleActionsField = (actionTypeRegistry, defaultValidationTimeout = DEFAULT_VALIDATION_TIMEOUT) => data => {
  let isCanceled = false;
  const promise = new Promise(resolve => {
    setTimeout(() => {
      if (isCanceled) {
        resolve();
      } else {
        resolve(validateRuleActionsField(actionTypeRegistry)(data));
      }
    }, defaultValidationTimeout);
  });
  promise.cancel = () => {
    isCanceled = true;
  };
  return promise;
};
exports.debouncedValidateRuleActionsField = debouncedValidateRuleActionsField;