"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isNoisy = exports.getTimeframeOptions = exports.getIsRulePreviewDisabled = exports.getHistogramConfig = void 0;
var _lodash = require("lodash");
var _charts = require("@elastic/charts");
var i18n = _interopRequireWildcard(require("./translations"));
var _utils = require("../../../../common/components/utils");
var _types = require("../../../pages/detection_engine/rules/types");
var _constants = require("../../../../../common/constants");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Determines whether or not to display noise warning.
 * Is considered noisy if alerts/hour rate > 1
 * @param hits Total query search hits
 * @param timeframe Range selected by user (last hour, day...)
 */
const isNoisy = (hits, timeframe) => {
  const oneHour = 1000 * 60 * 60;
  const durationInHours = Math.max((timeframe.timeframeEnd.valueOf() - timeframe.timeframeStart.valueOf()) / oneHour, 1.0);
  return hits / durationInHours > 1;
};

/**
 * Determines what timerange options to show.
 * Eql sequence queries tend to be slower, so decided
 * not to include the last month option.
 * @param ruleType
 */
exports.isNoisy = isNoisy;
const getTimeframeOptions = ruleType => {
  if (ruleType === 'eql') {
    return [{
      value: 'h',
      text: i18n.LAST_HOUR
    }, {
      value: 'd',
      text: i18n.LAST_DAY
    }];
  } else if (ruleType === 'threat_match') {
    return [{
      value: 'h',
      text: i18n.LAST_HOUR
    }, {
      value: 'd',
      text: i18n.LAST_DAY
    }, {
      value: 'w',
      text: i18n.LAST_WEEK
    }];
  } else if (ruleType === 'threshold') {
    return [{
      value: 'h',
      text: i18n.LAST_HOUR
    }];
  } else {
    return [{
      value: 'h',
      text: i18n.LAST_HOUR
    }, {
      value: 'd',
      text: i18n.LAST_DAY
    }, {
      value: 'M',
      text: i18n.LAST_MONTH
    }];
  }
};

/**
 * Config passed into elastic-charts settings.
 * @param to
 * @param from
 */
exports.getTimeframeOptions = getTimeframeOptions;
const getHistogramConfig = (to, from, showLegend = false) => {
  return {
    series: {
      xScaleType: _charts.ScaleType.Time,
      yScaleType: _charts.ScaleType.Linear,
      stackAccessors: ['g']
    },
    axis: {
      xTickFormatter: (0, _utils.histogramDateTimeFormatter)([to, from]),
      yTickFormatter: value => value.toLocaleString(),
      tickSize: 8
    },
    yAxisTitle: i18n.QUERY_GRAPH_COUNT,
    settings: {
      legendPosition: _charts.Position.Right,
      showLegend,
      showLegendExtra: showLegend,
      theme: {
        scales: {
          barsPadding: 0.08
        },
        chartMargins: {
          left: 0,
          right: 0,
          top: 0,
          bottom: 0
        },
        chartPaddings: {
          left: 0,
          right: 0,
          top: 0,
          bottom: 0
        }
      }
    },
    customHeight: 200
  };
};
exports.getHistogramConfig = getHistogramConfig;
const isNewTermsPreviewDisabled = newTermsFields => {
  return newTermsFields.length === 0 || newTermsFields.length > _constants.MAX_NUMBER_OF_NEW_TERMS_FIELDS;
};
const getIsRulePreviewDisabled = ({
  ruleType,
  isQueryBarValid,
  isThreatQueryBarValid,
  index,
  dataViewId,
  dataSourceType,
  threatIndex,
  threatMapping,
  machineLearningJobId,
  queryBar,
  newTermsFields
}) => {
  if (!isQueryBarValid || dataSourceType === _types.DataSourceType.DataView && !dataViewId || dataSourceType === _types.DataSourceType.IndexPatterns && index.length === 0) {
    return true;
  }
  if (ruleType === 'threat_match') {
    var _threatMapping$0$entr;
    if (!isThreatQueryBarValid || !threatIndex.length || !threatMapping) return true;
    if (!threatMapping.length || !((_threatMapping$0$entr = threatMapping[0].entries) !== null && _threatMapping$0$entr !== void 0 && _threatMapping$0$entr.length) || !threatMapping[0].entries[0].field || !threatMapping[0].entries[0].value) return true;
  }
  if (ruleType === 'machine_learning') {
    return machineLearningJobId.length === 0;
  }
  if (ruleType === 'eql' || ruleType === 'query' || ruleType === 'threshold') {
    return (0, _lodash.isEmpty)(queryBar.query.query) && (0, _lodash.isEmpty)(queryBar.filters);
  }
  if (ruleType === 'new_terms') {
    return isNewTermsPreviewDisabled(newTermsFields);
  }
  return false;
};
exports.getIsRulePreviewDisabled = getIsRulePreviewDisabled;