"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertRulesFilterToKQL = void 0;
var _kuery = require("../../../common/lib/kuery");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SEARCHABLE_RULE_PARAMS = ['alert.attributes.name', 'alert.attributes.params.index', 'alert.attributes.params.threat.tactic.id', 'alert.attributes.params.threat.tactic.name', 'alert.attributes.params.threat.technique.id', 'alert.attributes.params.threat.technique.name', 'alert.attributes.params.threat.technique.subtechnique.id', 'alert.attributes.params.threat.technique.subtechnique.name'];

/**
 * Convert rules filter options object to KQL query
 *
 * @param filterOptions desired filters (e.g. filter/sortField/sortOrder)
 *
 * @returns KQL string
 */
const convertRulesFilterToKQL = ({
  showCustomRules,
  showElasticRules,
  filter,
  tags,
  excludeRuleTypes = [],
  enabled
}) => {
  const filters = [];
  if (showCustomRules && showElasticRules) {
    // if both showCustomRules && showElasticRules selected we omit filter, as it includes all existing rules
  } else if (showElasticRules) {
    filters.push('alert.attributes.params.immutable: true');
  } else if (showCustomRules) {
    filters.push('alert.attributes.params.immutable: false');
  }
  if (enabled !== undefined) {
    filters.push(`alert.attributes.enabled: ${enabled ? 'true' : 'false'}`);
  }
  if (tags.length > 0) {
    filters.push(`alert.attributes.tags:(${tags.map(tag => `"${(0, _kuery.escapeKuery)(tag)}"`).join(' AND ')})`);
  }
  if (filter.length) {
    const searchQuery = SEARCHABLE_RULE_PARAMS.map(param => `${param}: "${(0, _kuery.escapeKuery)(filter)}"`).join(' OR ');
    filters.push(`(${searchQuery})`);
  }
  if (excludeRuleTypes.length) {
    filters.push(`NOT alert.attributes.params.type: (${excludeRuleTypes.map(ruleType => `"${(0, _kuery.escapeKuery)(ruleType)}"`).join(' OR ')})`);
  }
  return filters.join(' AND ');
};
exports.convertRulesFilterToKQL = convertRulesFilterToKQL;