"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useUpgradeSecurityPackages = exports.useIsUpgradingSecurityPackages = void 0;
var _reactQuery = require("@tanstack/react-query");
var _react = require("react");
var _constants = require("../../../../common/detection_engine/constants");
var _user_privileges = require("../../../common/components/user_privileges");
var _kibana = require("../../../common/lib/kibana");
var _use_bulk_install_fleet_packages_mutation = require("../api/hooks/use_bulk_install_fleet_packages_mutation");
var _use_install_fleet_package_mutation = require("../api/hooks/use_install_fleet_package_mutation");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Install or upgrade the security packages (endpoint and prebuilt rules)
 */
const useUpgradeSecurityPackages = () => {
  const context = (0, _kibana.useKibana)();
  const canAccessFleet = (0, _user_privileges.useUserPrivileges)().endpointPrivileges.canAccessFleet;
  const {
    mutate: bulkInstallFleetPackages
  } = (0, _use_bulk_install_fleet_packages_mutation.useBulkInstallFleetPackagesMutation)();
  const {
    mutate: installFleetPackage
  } = (0, _use_install_fleet_package_mutation.useInstallFleetPackageMutation)();
  (0, _react.useEffect)(() => {
    if (!canAccessFleet) {
      return;
    }
    (async () => {
      var _context$services$fle;
      // Make sure fleet is initialized first
      await ((_context$services$fle = context.services.fleet) === null || _context$services$fle === void 0 ? void 0 : _context$services$fle.isInitialized());

      // Always install the latest package if in dev env or snapshot build
      const prerelease = _kibana.KibanaServices.getKibanaVersion().includes('-SNAPSHOT') || _kibana.KibanaServices.getKibanaBranch() === 'main';
      const prebuiltRulesPackageVersion = _kibana.KibanaServices.getPrebuiltRulesPackageVersion();
      // ignore the response for now since we aren't notifying the user
      const packages = ['endpoint', _constants.PREBUILT_RULES_PACKAGE_NAME];

      // If `prebuiltRulesPackageVersion` is provided, try to install that version
      // Must be done as two separate requests as bulk API doesn't support versions
      if (prebuiltRulesPackageVersion != null) {
        installFleetPackage({
          packageName: _constants.PREBUILT_RULES_PACKAGE_NAME,
          packageVersion: prebuiltRulesPackageVersion,
          prerelease,
          force: true
        });
        packages.splice(packages.indexOf(_constants.PREBUILT_RULES_PACKAGE_NAME), 1);
      }

      // Note: if `prerelease:true` option is provided, endpoint package will also be installed as prerelease
      bulkInstallFleetPackages({
        packages,
        prerelease
      });
    })();
  }, [bulkInstallFleetPackages, canAccessFleet, context.services.fleet, installFleetPackage]);
};

/**
 * @returns true if the security packages are being installed or upgraded
 */
exports.useUpgradeSecurityPackages = useUpgradeSecurityPackages;
const useIsUpgradingSecurityPackages = () => {
  const isInstallingPackages = (0, _reactQuery.useIsMutating)({
    predicate: ({
      options
    }) => {
      const {
        mutationKey,
        variables
      } = options;

      // The mutation is bulk Fleet packages installation. Check if the packages include the prebuilt rules package
      if (mutationKey === _use_bulk_install_fleet_packages_mutation.BULK_INSTALL_FLEET_PACKAGES_MUTATION_KEY) {
        return variables.packages.includes(_constants.PREBUILT_RULES_PACKAGE_NAME);
      }

      // The mutation is single Fleet package installation. Check if the package is the prebuilt rules package
      if (mutationKey === _use_install_fleet_package_mutation.INSTALL_FLEET_PACKAGE_MUTATION_KEY) {
        return variables.packageName === _constants.PREBUILT_RULES_PACKAGE_NAME;
      }
      return false;
    }
  });
  return isInstallingPackages > 0;
};
exports.useIsUpgradingSecurityPackages = useIsUpgradingSecurityPackages;