"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useUpdateUrlParam = exports.useSyncGlobalQueryString = exports.useInitializeUrlParam = exports.useGlobalQueryString = void 0;
var _react = require("react");
var _fp = require("lodash/fp");
var _reactRedux = require("react-redux");
var _usePrevious = _interopRequireDefault(require("react-use/lib/usePrevious"));
var _rison = require("@kbn/rison");
var _helpers = require("./helpers");
var _use_selector = require("../../hooks/use_selector");
var _global_url_param = require("../../store/global_url_param");
var _use_route_spy = require("../route/use_route_spy");
var _links = require("../../links");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Adds urlParamKey and the initial value to redux store.
 *
 * Please call this hook at the highest possible level of the rendering tree.
 * So it is only called when the application starts instead of on every page.
 *
 * @param urlParamKey Must not change.
 * @param onInitialize Called once when initializing. It must not change.
 */
const useInitializeUrlParam = (urlParamKey, onInitialize) => {
  const dispatch = (0, _reactRedux.useDispatch)();
  const getInitialUrlParamValue = (0, _helpers.useGetInitialUrlParamValue)(urlParamKey);
  (0, _react.useEffect)(() => {
    const value = getInitialUrlParamValue();
    dispatch(_global_url_param.globalUrlParamActions.registerUrlParam({
      key: urlParamKey,
      initialValue: value
    }));

    // execute consumer initialization
    onInitialize(value);
    return () => {
      dispatch(_global_url_param.globalUrlParamActions.deregisterUrlParam({
        key: urlParamKey
      }));
    };
    // eslint-disable-next-line react-hooks/exhaustive-deps -- It must run only once when the application is initializing.
  }, []);
};

/**
 * Updates URL parameters in the url.
 *
 * Make sure to call `useInitializeUrlParam` before calling this function.
 */
exports.useInitializeUrlParam = useInitializeUrlParam;
const useUpdateUrlParam = urlParamKey => {
  const dispatch = (0, _reactRedux.useDispatch)();
  const updateUrlParam = (0, _react.useCallback)(value => {
    dispatch(_global_url_param.globalUrlParamActions.updateUrlParam({
      key: urlParamKey,
      value
    }));
  }, [dispatch, urlParamKey]);
  return updateUrlParam;
};
exports.useUpdateUrlParam = useUpdateUrlParam;
const useGlobalQueryString = () => {
  const globalUrlParam = (0, _use_selector.useShallowEqualSelector)(_global_url_param.globalUrlParamSelectors.selectGlobalUrlParam);
  const globalQueryString = (0, _react.useMemo)(() => {
    const encodedGlobalUrlParam = {};
    if (!globalUrlParam) {
      return '';
    }
    Object.keys(globalUrlParam).forEach(paramName => {
      const value = globalUrlParam[paramName];
      if (!value || typeof value === 'object' && (0, _fp.isEmpty)(value)) {
        return;
      }
      try {
        encodedGlobalUrlParam[paramName] = (0, _rison.encode)(value);
      } catch {
        // Just ignore parameters which unable to encode
      }
    });
    return (0, _helpers.encodeQueryString)((0, _fp.pickBy)(value => !(0, _fp.isEmpty)(value), encodedGlobalUrlParam));
  }, [globalUrlParam]);
  return globalQueryString;
};

/**
 * - It hides / shows the global query depending on the page.
 * - It updates the URL when globalUrlParam store updates.
 */
exports.useGlobalQueryString = useGlobalQueryString;
const useSyncGlobalQueryString = () => {
  const [{
    pageName
  }] = (0, _use_route_spy.useRouteSpy)();
  const globalUrlParam = (0, _use_selector.useShallowEqualSelector)(_global_url_param.globalUrlParamSelectors.selectGlobalUrlParam);
  const previousGlobalUrlParams = (0, _usePrevious.default)(globalUrlParam);
  const replaceUrlParams = (0, _helpers.useReplaceUrlParams)();
  (0, _react.useEffect)(() => {
    var _getLinkInfo;
    const linkInfo = (_getLinkInfo = (0, _links.getLinkInfo)(pageName)) !== null && _getLinkInfo !== void 0 ? _getLinkInfo : {
      skipUrlState: true
    };
    const paramsToUpdate = {
      ...globalUrlParam
    };
    if (linkInfo.skipUrlState) {
      Object.keys(paramsToUpdate).forEach(key => {
        paramsToUpdate[key] = null;
      });
    }

    // Url params that got deleted from GlobalUrlParams
    const unregisteredKeys = (0, _fp.difference)(Object.keys(previousGlobalUrlParams !== null && previousGlobalUrlParams !== void 0 ? previousGlobalUrlParams : {}), Object.keys(globalUrlParam));

    // Delete unregistered Url params
    unregisteredKeys.forEach(key => {
      paramsToUpdate[key] = null;
    });
    if (Object.keys(paramsToUpdate).length > 0) {
      replaceUrlParams(paramsToUpdate);
    }
  }, [previousGlobalUrlParams, globalUrlParam, pageName, replaceUrlParams]);
};
exports.useSyncGlobalQueryString = useSyncGlobalQueryString;