"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useLinkExists = exports.useAppLinks = exports.updateAppLinks = exports.subscribeAppLinks = exports.needsUrlState = exports.hasCapabilities = exports.getLinksWithHiddenTimeline = exports.getLinkInfo = exports.getAncestorLinksInfo = void 0;
var _lodash = require("lodash");
var _react = require("react");
var _rxjs = require("rxjs");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * App links updater, it keeps the value of the app links in sync with all application.
 * It can be updated using `updateAppLinks` or `excludeAppLink`
 * Read it using `subscribeAppLinks` or `useAppLinks` hook.
 */
const appLinksUpdater$ = new _rxjs.BehaviorSubject({
  links: [],
  // stores the appLinkItems recursive hierarchy
  normalizedLinks: {} // stores a flatten normalized object for direct id access
});

const getAppLinksValue = () => appLinksUpdater$.getValue().links;
const getNormalizedLinksValue = () => appLinksUpdater$.getValue().normalizedLinks;

/**
 * Subscribes to the updater to get the app links updates
 */
const subscribeAppLinks = onChange => appLinksUpdater$.subscribe(({
  links
}) => onChange(links));

/**
 * Hook to get the app links updated value
 */
exports.subscribeAppLinks = subscribeAppLinks;
const useAppLinks = () => {
  const [appLinks, setAppLinks] = (0, _react.useState)(getAppLinksValue);
  (0, _react.useEffect)(() => {
    const linksSubscription = subscribeAppLinks(newAppLinks => {
      setAppLinks(newAppLinks);
    });
    return () => linksSubscription.unsubscribe();
  }, []);
  return appLinks;
};

/**
 * Hook to check if a link exists in the application links,
 * It can be used to know if a link access is authorized.
 */
exports.useAppLinks = useAppLinks;
const useLinkExists = id => {
  const [linkExists, setLinkExists] = (0, _react.useState)(!!getNormalizedLink(id));
  (0, _react.useEffect)(() => {
    const linksSubscription = subscribeAppLinks(() => {
      setLinkExists(!!getNormalizedLink(id));
    });
    return () => linksSubscription.unsubscribe();
  }, [id]);
  return linkExists;
};

/**
 * Updates the app links applying the filter by permissions
 */
exports.useLinkExists = useLinkExists;
const updateAppLinks = (appLinksToUpdate, linksPermissions) => {
  const filteredAppLinks = getFilteredAppLinks(appLinksToUpdate, linksPermissions);
  appLinksUpdater$.next({
    links: Object.freeze(filteredAppLinks),
    normalizedLinks: Object.freeze(getNormalizedLinks(filteredAppLinks))
  });
};

/**
 * Returns the `LinkInfo` from a link id parameter
 */
exports.updateAppLinks = updateAppLinks;
const getLinkInfo = id => {
  const normalizedLink = getNormalizedLink(id);
  if (!normalizedLink) {
    return undefined;
  }
  // discards the parentId and creates the linkInfo copy.
  const {
    parentId,
    ...linkInfo
  } = normalizedLink;
  return linkInfo;
};

/**
 * Returns the `LinkInfo` of all the ancestors to the parameter id link, also included.
 */
exports.getLinkInfo = getLinkInfo;
const getAncestorLinksInfo = id => {
  const ancestors = [];
  let currentId = id;
  while (currentId) {
    const normalizedLink = getNormalizedLink(currentId);
    if (normalizedLink) {
      const {
        parentId,
        ...linkInfo
      } = normalizedLink;
      ancestors.push(linkInfo);
      currentId = parentId;
    } else {
      currentId = undefined;
    }
  }
  return ancestors.reverse();
};

/**
 * Returns `true` if the links needs to carry the application state in the url.
 * Defaults to `true` if the `skipUrlState` property of the `LinkItem` is `undefined`.
 */
exports.getAncestorLinksInfo = getAncestorLinksInfo;
const needsUrlState = id => {
  var _getNormalizedLink;
  return !((_getNormalizedLink = getNormalizedLink(id)) !== null && _getNormalizedLink !== void 0 && _getNormalizedLink.skipUrlState);
};

// Internal functions

/**
 * Creates the `NormalizedLinks` structure from a `LinkItem` array
 */
exports.needsUrlState = needsUrlState;
const getNormalizedLinks = (currentLinks, parentId) => {
  return currentLinks.reduce((normalized, {
    links,
    ...currentLink
  }) => {
    normalized[currentLink.id] = {
      ...currentLink,
      parentId
    };
    if (links && links.length > 0) {
      Object.assign(normalized, getNormalizedLinks(links, currentLink.id));
    }
    return normalized;
  }, {});
};
const getNormalizedLink = id => getNormalizedLinksValue()[id];
const getFilteredAppLinks = (appLinkToFilter, linksPermissions) => appLinkToFilter.reduce((acc, {
  links,
  ...appLink
}) => {
  if (!isLinkAllowed(appLink, linksPermissions)) {
    return acc;
  }
  if (links) {
    const childrenLinks = getFilteredAppLinks(links, linksPermissions);
    if (childrenLinks.length > 0) {
      acc.push({
        ...appLink,
        links: childrenLinks
      });
    } else {
      acc.push(appLink);
    }
  } else {
    acc.push(appLink);
  }
  return acc;
}, []);

/**
 * The format of defining features supports OR and AND mechanism. To specify features in an OR fashion
 * they can be defined in a single level array like: [requiredFeature1, requiredFeature2]. If either of these features
 * is satisfied the links would be included. To require that the features be AND'd together a second level array
 * can be specified: [feature1, [feature2, feature3]] this would result in feature1 || (feature2 && feature3).
 *
 * The final format is to specify a single feature, this would be like: features: feature1, which is the same as
 * features: [feature1]
 */

// It checks if the user has at least one of the link capabilities needed
const hasCapabilities = (linkCapabilities, userCapabilities) => {
  if (!(0, _lodash.isArray)(linkCapabilities)) {
    return !!(0, _lodash.get)(userCapabilities, linkCapabilities, false);
  } else {
    return linkCapabilities.some(linkCapabilityKeyOr => {
      if ((0, _lodash.isArray)(linkCapabilityKeyOr)) {
        return linkCapabilityKeyOr.every(linkCapabilityKeyAnd => (0, _lodash.get)(userCapabilities, linkCapabilityKeyAnd, false));
      }
      return (0, _lodash.get)(userCapabilities, linkCapabilityKeyOr, false);
    });
  }
};
exports.hasCapabilities = hasCapabilities;
const isLinkAllowed = (link, {
  license,
  experimentalFeatures,
  capabilities
}) => {
  var _link$licenseType;
  const linkLicenseType = (_link$licenseType = link.licenseType) !== null && _link$licenseType !== void 0 ? _link$licenseType : 'basic';
  if (license) {
    if (!license.hasAtLeast(linkLicenseType)) {
      return false;
    }
  } else if (linkLicenseType !== 'basic') {
    return false;
  }
  if (link.hideWhenExperimentalKey && experimentalFeatures[link.hideWhenExperimentalKey]) {
    return false;
  }
  if (link.experimentalKey && !experimentalFeatures[link.experimentalKey]) {
    return false;
  }
  if (link.capabilities && !hasCapabilities(link.capabilities, capabilities)) {
    return false;
  }
  return true;
};
const getLinksWithHiddenTimeline = () => {
  return Object.values(getNormalizedLinksValue()).filter(link => link.hideTimeline);
};
exports.getLinksWithHiddenTimeline = getLinksWithHiddenTimeline;