"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useInsightDataProviders = void 0;
var _react = require("react");
var _esQuery = require("@kbn/es-query");
var _common = require("@kbn/timelines-plugin/common");
var _replace_params_query = require("./replace_params_query");
var _data_provider = require("../../../../../timelines/components/timeline/data_providers/data_provider");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const dataProviderQueryType = type => {
  if (type === _esQuery.FILTERS.EXISTS) {
    return _data_provider.EXISTS_OPERATOR;
  } else if (type === _esQuery.FILTERS.PHRASES) {
    return _data_provider.IS_ONE_OF_OPERATOR;
  } else {
    return _data_provider.IS_OPERATOR;
  }
};
const filterStub = {
  $state: {
    store: _esQuery.FilterStateStore.APP_STATE
  },
  meta: {
    disabled: false,
    negate: false,
    alias: null,
    index: undefined
  }
};
const dataProviderStub = {
  and: [],
  enabled: true,
  id: '',
  name: '',
  excluded: false,
  kqlQuery: '',
  type: _common.DataProviderType.default,
  queryMatch: {
    field: '',
    value: '',
    operator: _data_provider.EXISTS_OPERATOR
  }
};
const buildDataProviders = (providers, alertData) => {
  return providers.map(innerProvider => {
    return innerProvider.reduce((prev, next, index) => {
      const {
        field,
        value,
        excluded,
        queryType
      } = next;
      const {
        result,
        matchedBrackets
      } = (0, _replace_params_query.replaceParamsQuery)(value, alertData);
      const isTemplate = !alertData && matchedBrackets;
      if (index === 0) {
        return {
          and: [],
          enabled: true,
          id: JSON.stringify(field + value),
          name: field,
          excluded,
          kqlQuery: '',
          type: isTemplate ? _common.DataProviderType.template : _common.DataProviderType.default,
          queryMatch: {
            field,
            value: result,
            operator: dataProviderQueryType(queryType)
          }
        };
      } else {
        const newProvider = {
          enabled: true,
          id: JSON.stringify(field + value),
          name: field,
          excluded,
          kqlQuery: '',
          type: isTemplate ? _common.DataProviderType.template : _common.DataProviderType.default,
          queryMatch: {
            field,
            value: result,
            operator: dataProviderQueryType(queryType)
          }
        };
        prev.and.push(newProvider);
      }
      return prev;
    }, dataProviderStub);
  });
};
const buildPrimitiveFilter = provider => {
  const baseFilter = {
    ...filterStub,
    meta: {
      ...filterStub.meta,
      negate: provider.excluded,
      type: provider.queryType
    }
  };
  if (provider.queryType === _esQuery.FILTERS.EXISTS) {
    return {
      ...baseFilter,
      meta: {
        ...baseFilter.meta,
        params: undefined,
        value: 'exists'
      },
      query: {
        exists: {
          field: provider.field
        }
      }
    };
  } else if (provider.queryType === _esQuery.FILTERS.PHRASES) {
    var _values;
    let values = JSON.parse(String(provider.value));
    if (provider.valueType === 'number') {
      values = values.map(Number);
    } else if (provider.valueType === 'boolean') {
      values = values.map(Boolean);
    }
    return {
      ...baseFilter,
      meta: {
        ...baseFilter.meta
      },
      query: {
        bool: {
          minimum_should_match: 1,
          should: (_values = values) === null || _values === void 0 ? void 0 : _values.map(param => ({
            match_phrase: {
              [provider.field]: param
            }
          }))
        }
      }
    };
  } else if (provider.queryType === _esQuery.FILTERS.PHRASE) {
    var _provider$value;
    return {
      ...baseFilter,
      meta: {
        ...baseFilter.meta,
        params: {
          query: provider.value
        },
        value: undefined
      },
      query: {
        match_phrase: {
          [provider.field]: (_provider$value = provider.value) !== null && _provider$value !== void 0 ? _provider$value : ''
        }
      }
    };
  } else if (provider.queryType === _esQuery.FILTERS.RANGE) {
    let gte;
    let lt;
    try {
      const input = JSON.parse(String(provider.value));
      gte = input.gte;
      lt = input.lt;
    } catch {
      gte = '';
      lt = '';
    }
    const params = {
      gte,
      lt
    };
    return {
      ...baseFilter,
      meta: {
        ...baseFilter.meta,
        params
      },
      query: {
        range: {
          [provider.field]: params
        }
      }
    };
  } else {
    return baseFilter;
  }
};
const buildFiltersFromInsightProviders = (providers, alertData) => {
  const filters = [];
  for (let index = 0; index < providers.length; index++) {
    const provider = providers[index];
    if (provider.length > 1) {
      // Only support 1 level of nesting currently
      const innerProviders = provider.map(innerProvider => {
        return buildPrimitiveFilter(innerProvider);
      });
      const combinedFilter = {
        $state: {
          store: _esQuery.FilterStateStore.APP_STATE
        },
        meta: {
          type: _esQuery.FILTERS.COMBINED,
          relation: _esQuery.BooleanRelation.AND,
          params: innerProviders,
          index: undefined,
          disabled: false,
          negate: false
        }
      };
      filters.push(combinedFilter);
    } else {
      const baseProvider = provider[0];
      const baseFilter = buildPrimitiveFilter(baseProvider);
      filters.push(baseFilter);
    }
  }
  return filters;
};
const useInsightDataProviders = ({
  providers,
  alertData
}) => {
  const providersContainRangeQuery = (0, _react.useMemo)(() => {
    return providers.some(innerProvider => {
      return innerProvider.some(provider => provider.queryType === 'range');
    });
  }, [providers]);
  const dataProviders = (0, _react.useMemo)(() => {
    if (providersContainRangeQuery) {
      return [];
    } else {
      return buildDataProviders(providers, alertData);
    }
  }, [alertData, providers, providersContainRangeQuery]);
  const filters = (0, _react.useMemo)(() => {
    if (!providersContainRangeQuery) {
      return [];
    } else {
      return buildFiltersFromInsightProviders(providers, alertData);
    }
  }, [providersContainRangeQuery, providers, alertData]);
  return {
    dataProviders,
    filters
  };
};
exports.useInsightDataProviders = useInsightDataProviders;