"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.timelineDataToEnrichment = exports.parseExistingEnrichments = exports.isInvestigationTimeEnrichment = exports.getShimmedIndicatorValue = exports.getFirstSeen = exports.getEnrichmentValue = exports.getEnrichmentIdentifiers = exports.getEnrichmentFields = exports.filterDuplicateEnrichments = void 0;
var _lodash = require("lodash");
var _field_formatters = require("../../../../../common/utils/field_formatters");
var _constants = require("../../../../../common/constants");
var _constants2 = require("../../../../../common/cti/constants");
var _cti = require("../../../../../common/search_strategy/security_solution/cti");
var _data_retrieval = require("../../../../../common/utils/data_retrieval");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const isInvestigationTimeEnrichment = type => type === _constants2.ENRICHMENT_TYPES.InvestigationTime;
exports.isInvestigationTimeEnrichment = isInvestigationTimeEnrichment;
const parseExistingEnrichments = data => {
  const threatIndicatorField = data.find(({
    field,
    originalValue
  }) => field === _constants.ENRICHMENT_DESTINATION_PATH && originalValue);
  if (!threatIndicatorField) {
    return [];
  }
  const {
    originalValue
  } = threatIndicatorField;
  const enrichmentStrings = Array.isArray(originalValue) ? originalValue : [originalValue];
  return enrichmentStrings.reduce((enrichments, enrichmentString) => {
    try {
      const enrichment = (0, _field_formatters.getDataFromFieldsHits)(JSON.parse(enrichmentString));
      enrichments.push(enrichment);
    } catch (e) {
      // omit failed parse
    }
    return enrichments;
  }, []);
};
exports.parseExistingEnrichments = parseExistingEnrichments;
const timelineDataToEnrichment = data => data.reduce((acc, item) => {
  acc[item.field] = item.originalValue;
  return acc;
}, {});
exports.timelineDataToEnrichment = timelineDataToEnrichment;
const getEnrichmentValue = (enrichment, field) => (0, _data_retrieval.getFirstElement)(enrichment[field]);

/**
 * These fields (e.g. 'indicator.ip') may be in one of three places depending on whether it's:
 *   * a queried, legacy filebeat indicator ('threatintel.indicator.ip')
 *   * a queried, ECS 1.11 filebeat indicator ('threat.indicator.ip')
 *   * an existing indicator from an enriched alert ('indicator.ip')
 */
exports.getEnrichmentValue = getEnrichmentValue;
const getShimmedIndicatorValue = (enrichment, field) => getEnrichmentValue(enrichment, field) || getEnrichmentValue(enrichment, `threatintel.${field}`) || getEnrichmentValue(enrichment, `threat.${field}`);
exports.getShimmedIndicatorValue = getShimmedIndicatorValue;
const getEnrichmentIdentifiers = enrichment => ({
  id: getEnrichmentValue(enrichment, _constants2.MATCHED_ID),
  field: getEnrichmentValue(enrichment, _constants2.MATCHED_FIELD),
  value: getEnrichmentValue(enrichment, _constants2.MATCHED_ATOMIC),
  type: getEnrichmentValue(enrichment, _constants2.MATCHED_TYPE),
  feedName: getShimmedIndicatorValue(enrichment, _constants2.FEED_NAME)
});
exports.getEnrichmentIdentifiers = getEnrichmentIdentifiers;
const buildEnrichmentId = enrichment => {
  const {
    id,
    field
  } = getEnrichmentIdentifiers(enrichment);
  return `${id}${field}`;
};

/**
 * This function receives an array of enrichments and removes
 * investigation-time enrichments if that exact indicator already exists
 * elsewhere in the list.
 *
 * @param enrichments {@type CtiEnrichment[]}
 */
const filterDuplicateEnrichments = enrichments => {
  if (enrichments.length < 2) {
    return enrichments;
  }
  const enrichmentsById = (0, _lodash.groupBy)(enrichments, buildEnrichmentId);
  return Object.values(enrichmentsById).map(enrichmentGroup => {
    var _enrichmentGroup$find;
    return (_enrichmentGroup$find = enrichmentGroup.find(enrichment => !isInvestigationTimeEnrichment(getEnrichmentValue(enrichment, _constants2.MATCHED_TYPE)))) !== null && _enrichmentGroup$find !== void 0 ? _enrichmentGroup$find : enrichmentGroup[0];
  });
};
exports.filterDuplicateEnrichments = filterDuplicateEnrichments;
const getEnrichmentFields = items => items.reduce((fields, item) => {
  if ((0, _cti.isValidEventField)(item.field)) {
    const value = (0, _data_retrieval.getFirstElement)(item.originalValue);
    if (value) {
      return {
        ...fields,
        [item.field]: value
      };
    }
  }
  return fields;
}, {});
exports.getEnrichmentFields = getEnrichmentFields;
const getFirstSeen = enrichment => {
  const firstSeenValue = getShimmedIndicatorValue(enrichment, _constants2.FIRST_SEEN);
  const firstSeenDate = Date.parse(firstSeenValue !== null && firstSeenValue !== void 0 ? firstSeenValue : 'no date');
  return Number.isInteger(firstSeenDate) ? firstSeenDate : new Date(-1).valueOf();
};
exports.getFirstSeen = getFirstSeen;