"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isValidDataProviderField = exports.getDataProvider = exports.createDataProviders = void 0;
var _securitysolutionTGrid = require("@kbn/securitysolution-t-grid");
var _fp = require("lodash/fp");
var _constants = require("../../../common/cti/constants");
var _types = require("../../../common/types");
var _ip = require("../../explore/network/components/ip");
var _helpers = require("../../explore/network/components/port/helpers");
var _duration = require("../../timelines/components/duration");
var _bytes = require("../../timelines/components/timeline/body/renderers/bytes");
var _constants2 = require("../../timelines/components/timeline/body/renderers/constants");
var _utils = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getDataProvider = ({
  field,
  id,
  value,
  operator = _types.IS_OPERATOR,
  excluded = false
}) => ({
  and: [],
  enabled: true,
  id: (0, _securitysolutionTGrid.escapeDataProviderId)(id),
  name: field,
  excluded,
  kqlQuery: '',
  queryMatch: {
    field,
    value,
    operator
  }
});
exports.getDataProvider = getDataProvider;
const createDataProviders = ({
  contextId,
  eventId,
  field,
  fieldFormat,
  fieldType,
  values,
  sourceParamType,
  negate
}) => {
  if (field == null) return null;
  const arrayValues = Array.isArray(values) ? values.length > 0 ? values : [null] : [values];
  return arrayValues.reduce((dataProviders, rawValue, index) => {
    let id = '';
    const value = rawValue != null ? rawValue.toString() : rawValue;
    const appendedUniqueId = `${contextId}${eventId ? `-${eventId}` : ''}-${field}-${index}${value ? `-${value}` : ''}`;
    if (!isValidDataProviderField(field, fieldType)) {
      return dataProviders;
    }
    if ((0, _utils.isCountField)(fieldType, sourceParamType)) {
      id = `value-count-data-provider-${contextId}-${field}`;
      dataProviders.push(getDataProvider({
        field,
        id,
        value: '',
        excluded: false,
        operator: _types.EXISTS_OPERATOR
      }));
      return dataProviders;
    }
    if (value == null) {
      id = `empty-value-${appendedUniqueId}`;
      dataProviders.push(getDataProvider({
        field,
        id,
        value: '',
        excluded: true,
        operator: _types.EXISTS_OPERATOR
      }));
      return dataProviders;
    }
    if (fieldType === _ip.IP_FIELD_TYPE) {
      id = `formatted-ip-data-provider-${contextId}-${field}-${value}${eventId ? `-${eventId}` : ''}`;
      if ((0, _fp.isString)(value) && !(0, _fp.isEmpty)(value)) {
        let addresses = value;
        try {
          addresses = JSON.parse(value);
        } catch (_) {
          // Default to keeping the existing string value
        }
        if ((0, _fp.isArray)(addresses)) {
          addresses.forEach(ip => dataProviders.push(getDataProvider({
            field,
            id,
            value: ip
          })));
        } else {
          dataProviders.push(getDataProvider({
            field,
            id,
            value: addresses
          }));
        }
        return dataProviders;
      }
    }
    id = getIdForField({
      field,
      fieldFormat,
      appendedUniqueId,
      value
    });
    dataProviders.push(getDataProvider({
      field,
      id,
      value,
      excluded: negate
    }));
    return dataProviders;
  }, []);
};
exports.createDataProviders = createDataProviders;
const isValidDataProviderField = (fieldName, fieldType) => fieldType !== _constants2.GEO_FIELD_TYPE && fieldName !== _constants2.MESSAGE_FIELD_NAME;
exports.isValidDataProviderField = isValidDataProviderField;
const getIdForField = ({
  field,
  fieldFormat,
  appendedUniqueId,
  value
}) => {
  let id;
  if (_helpers.PORT_NAMES.some(portName => field === portName)) {
    id = `port-default-${appendedUniqueId}`;
  } else if (field === _duration.EVENT_DURATION_FIELD_NAME) {
    id = `duration-default-${appendedUniqueId}`;
  } else if (fieldFormat === _bytes.BYTES_FORMAT) {
    id = `bytes-default-${appendedUniqueId}`;
  } else if (field === _constants2.SIGNAL_STATUS_FIELD_NAME) {
    id = `alert-field-default-${appendedUniqueId}`;
  } else if ([_constants2.RULE_REFERENCE_FIELD_NAME, _constants2.REFERENCE_URL_FIELD_NAME, _constants2.EVENT_URL_FIELD_NAME, _constants.INDICATOR_REFERENCE, _constants2.SIGNAL_RULE_NAME_FIELD_NAME, _constants2.EVENT_MODULE_FIELD_NAME].includes(field)) {
    id = `event-field-default-${appendedUniqueId}-${value}`;
  } else {
    id = `event-field-default-${appendedUniqueId}`;
  }
  return id;
};