"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.startMetadataTransforms = startMetadataTransforms;
exports.stopMetadataTransforms = stopMetadataTransforms;
exports.waitForMetadataTransformsReady = waitForMetadataTransformsReady;
var _constants = require("../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function waitForMetadataTransformsReady(esClient) {
  await waitFor(() => areMetadataTransformsReady(esClient));
}
async function stopMetadataTransforms(esClient) {
  const transformIds = await getMetadataTransformIds(esClient);
  await Promise.all(transformIds.map(transformId => esClient.transform.stopTransform({
    transform_id: transformId,
    force: true,
    wait_for_completion: true,
    allow_no_match: true
  })));
}
async function startMetadataTransforms(esClient,
// agentIds to wait for
agentIds) {
  const transformIds = await getMetadataTransformIds(esClient);
  const currentTransformId = transformIds.find(transformId => transformId.startsWith(_constants.metadataTransformPrefix));
  const unitedTransformId = transformIds.find(transformId => transformId.startsWith(_constants.METADATA_UNITED_TRANSFORM));
  if (!currentTransformId || !unitedTransformId) {
    // eslint-disable-next-line no-console
    console.warn('metadata transforms not found, skipping transform start');
    return;
  }
  try {
    await esClient.transform.startTransform({
      transform_id: currentTransformId
    });
  } catch (err) {
    // ignore if transform already started
    if (err.statusCode !== 409) {
      throw err;
    }
  }
  await waitForCurrentMetdataDocs(esClient, agentIds);
  try {
    await esClient.transform.startTransform({
      transform_id: unitedTransformId
    });
  } catch (err) {
    // ignore if transform already started
    if (err.statusCode !== 409) {
      throw err;
    }
  }
}
async function getMetadataTransformStats(esClient) {
  return (await esClient.transform.getTransformStats({
    transform_id: _constants.METADATA_TRANSFORMS_PATTERN,
    allow_no_match: true
  })).transforms;
}
async function getMetadataTransformIds(esClient) {
  return (await getMetadataTransformStats(esClient)).map(transform => transform.id);
}
async function areMetadataTransformsReady(esClient) {
  const transforms = await getMetadataTransformStats(esClient);
  return !transforms.some(
  // TODO TransformGetTransformStatsTransformStats type needs to be updated to include health
  transform => {
    var _transform$health;
    return (transform === null || transform === void 0 ? void 0 : (_transform$health = transform.health) === null || _transform$health === void 0 ? void 0 : _transform$health.status) !== 'green';
  });
}
async function waitForCurrentMetdataDocs(esClient, agentIds) {
  var _agentIds$length;
  const query = agentIds.length ? {
    bool: {
      filter: [{
        terms: {
          'agent.id': agentIds
        }
      }]
    }
  } : {
    match_all: {}
  };
  const size = (_agentIds$length = agentIds.length) !== null && _agentIds$length !== void 0 ? _agentIds$length : 1;
  await waitFor(async () => (await esClient.search({
    index: _constants.metadataCurrentIndexPattern,
    query,
    size,
    rest_total_hits_as_int: true
  })).hits.total === size);
}
async function waitFor(cb, interval = 20000, maxAttempts = 6) {
  let attempts = 0;
  let isReady = false;
  while (!isReady) {
    await new Promise(res => setTimeout(() => res(''), interval));
    isReady = await cb();
    attempts++;
    if (attempts > maxAttempts) {
      return;
    }
  }
}