"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.determineIfValueCanUpdate = exports.determineDiffOutcome = exports.ThreeWayDiffOutcome = void 0;
var _lodash = require("lodash");
var _three_way_diff = require("./three_way_diff");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
/**
 * Result of comparing three versions of a value against each other.
 * Defines 5 typical combinations of 3 versions of a value.
 */
let ThreeWayDiffOutcome;
exports.ThreeWayDiffOutcome = ThreeWayDiffOutcome;
(function (ThreeWayDiffOutcome) {
  ThreeWayDiffOutcome["StockValueNoUpdate"] = "BASE=A, CURRENT=A, TARGET=A";
  ThreeWayDiffOutcome["StockValueCanUpdate"] = "BASE=A, CURRENT=A, TARGET=B";
  ThreeWayDiffOutcome["CustomizedValueNoUpdate"] = "BASE=A, CURRENT=B, TARGET=A";
  ThreeWayDiffOutcome["CustomizedValueSameUpdate"] = "BASE=A, CURRENT=B, TARGET=B";
  ThreeWayDiffOutcome["CustomizedValueCanUpdate"] = "BASE=A, CURRENT=B, TARGET=C";
})(ThreeWayDiffOutcome || (exports.ThreeWayDiffOutcome = ThreeWayDiffOutcome = {}));
const determineDiffOutcome = (baseVersion, currentVersion, targetVersion) => {
  const baseEqlCurrent = (0, _lodash.isEqual)(baseVersion, currentVersion);
  const baseEqlTarget = (0, _lodash.isEqual)(baseVersion, targetVersion);
  const currentEqlTarget = (0, _lodash.isEqual)(currentVersion, targetVersion);
  if (baseVersion === _three_way_diff.MissingVersion) {
    /**
     * We couldn't find the base version of the rule in the package so further
     * version comparison is not possible. We assume that the rule is not
     * customized and the value can be updated if there's an update.
     */
    return currentEqlTarget ? ThreeWayDiffOutcome.StockValueNoUpdate : ThreeWayDiffOutcome.StockValueCanUpdate;
  }
  if (baseEqlCurrent) {
    return currentEqlTarget ? ThreeWayDiffOutcome.StockValueNoUpdate : ThreeWayDiffOutcome.StockValueCanUpdate;
  }
  if (baseEqlTarget) {
    return ThreeWayDiffOutcome.CustomizedValueNoUpdate;
  }
  return currentEqlTarget ? ThreeWayDiffOutcome.CustomizedValueSameUpdate : ThreeWayDiffOutcome.CustomizedValueCanUpdate;
};
exports.determineDiffOutcome = determineDiffOutcome;
const determineIfValueCanUpdate = diffCase => {
  return diffCase === ThreeWayDiffOutcome.StockValueCanUpdate || diffCase === ThreeWayDiffOutcome.CustomizedValueCanUpdate;
};
exports.determineIfValueCanUpdate = determineIfValueCanUpdate;