"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildSavedObject = buildSavedObject;
var _lodash = require("lodash");
var _hydrate_index_pattern = require("./hydrate_index_pattern");
var _initialize_saved_object = require("./initialize_saved_object");
var _serialize_saved_object = require("./serialize_saved_object");
var _apply_es_resp = require("./apply_es_resp");
var _save_saved_object = require("./save_saved_object");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const applyDecorators = (object, config, decorators) => {
  decorators.forEach(decorator => {
    decorator.decorateConfig(config);
    decorator.decorateObject(object);
  });
};
function buildSavedObject(savedObject, config, services, decorators = []) {
  applyDecorators(savedObject, config, decorators);
  const {
    dataViews,
    savedObjectsClient
  } = services;
  // type name for this object, used as the ES-type
  const esType = config.type || '';
  savedObject.getDisplayName = () => esType;

  // NOTE: this.type (not set in this file, but somewhere else) is the sub type, e.g. 'area' or
  // 'data table', while esType is the more generic type - e.g. 'visualization' or 'saved search'.
  savedObject.getEsType = () => esType;

  /**
   * Flips to true during a save operation, and back to false once the save operation
   * completes.
   * @type {boolean}
   */
  savedObject.isSaving = false;
  savedObject.defaults = config.defaults || {};
  // optional search source which this object configures
  savedObject.searchSource = config.searchSource ? services.search.searchSource.createEmpty() : undefined;
  // the id of the document
  savedObject.id = config.id || void 0;
  // the migration version of the document, should only be set on imports
  savedObject.migrationVersion = config.migrationVersion;
  // Whether to create a copy when the object is saved. This should eventually go away
  // in favor of a better rename/save flow.
  savedObject.copyOnSave = false;

  /**
   * After creation or fetching from ES, ensure that the searchSources index indexPattern
   * is an bonafide IndexPattern object.
   *
   * @return {Promise<IndexPattern | null>}
   */
  savedObject.hydrateIndexPattern = id => (0, _hydrate_index_pattern.hydrateIndexPattern)(id || '', savedObject, dataViews, config);
  /**
   * Asynchronously initialize this object - will only run
   * once even if called multiple times.
   *
   * @return {Promise}
   * @resolved {SavedObject}
   */
  savedObject.init = (0, _lodash.once)(() => (0, _initialize_saved_object.intializeSavedObject)(savedObject, savedObjectsClient, config));
  savedObject.applyESResp = resp => (0, _apply_es_resp.applyESResp)(resp, savedObject, config, services);

  /**
   * Serialize this object
   * @return {Object}
   */
  savedObject._serialize = () => (0, _serialize_saved_object.serializeSavedObject)(savedObject, config);

  /**
   * Returns true if the object's original title has been changed. New objects return false.
   * @return {boolean}
   */
  savedObject.isTitleChanged = () => savedObject._source && savedObject._source.title !== savedObject.title;
  savedObject.creationOpts = (opts = {}) => ({
    id: savedObject.id,
    migrationVersion: savedObject.migrationVersion,
    ...opts
  });
  savedObject.save = async opts => {
    try {
      const result = await (0, _save_saved_object.saveSavedObject)(savedObject, config, opts, services);
      return Promise.resolve(result);
    } catch (e) {
      return Promise.reject(e);
    }
  };
  savedObject.destroy = () => {};

  /**
   * Delete this object from Elasticsearch
   * @return {promise}
   */
  savedObject.delete = () => {
    if (!savedObject.id) {
      return Promise.reject(new Error('Deleting a saved Object requires type and id'));
    }
    return savedObjectsClient.delete(esType, savedObject.id);
  };
}