"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ReportingAPIClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _i18n = require("@kbn/i18n");
var _moment = _interopRequireDefault(require("moment"));
var _queryString = require("query-string");
var _rison = _interopRequireDefault(require("@kbn/rison"));
var _build_kibana_path = require("../../../common/build_kibana_path");
var _constants = require("../../../common/constants");
var _job_completion_notifications = require("../../notifier/job_completion_notifications");
var _job = require("../job");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Client class for interacting with Reporting APIs
 * @implements IReportingAPI
 */
class ReportingAPIClient {
  constructor(http, uiSettings, kibanaVersion) {
    (0, _defineProperty2.default)(this, "http", void 0);
    (0, _defineProperty2.default)(this, "getManagementLink", () => this.http.basePath.prepend(_constants.REPORTING_MANAGEMENT_HOME));
    (0, _defineProperty2.default)(this, "getDownloadLink", jobId => this.http.basePath.prepend(`${_constants.API_LIST_URL}/download/${jobId}`));
    (0, _defineProperty2.default)(this, "getServerBasePath", () => this.http.basePath.serverBasePath);
    this.uiSettings = uiSettings;
    this.kibanaVersion = kibanaVersion;
    this.http = http;
  }
  getKibanaAppHref(job) {
    const searchParams = (0, _queryString.stringify)({
      jobId: job.id
    });
    const path = (0, _build_kibana_path.buildKibanaPath)({
      basePath: this.http.basePath.serverBasePath,
      spaceId: job.spaceId,
      appPath: (0, _constants.getRedirectAppPath)()
    });
    const href = `${path}?${searchParams}`;
    return href;
  }
  getReportURL(jobId) {
    const apiBaseUrl = this.http.basePath.prepend(_constants.API_LIST_URL);
    const downloadLink = `${apiBaseUrl}/download/${jobId}`;
    return downloadLink;
  }
  downloadReport(jobId) {
    const location = this.getReportURL(jobId);
    window.open(location);
  }
  async deleteReport(jobId) {
    return await this.http.delete(`${_constants.API_LIST_URL}/delete/${jobId}`, {
      asSystemRequest: true
    });
  }
  async list(page = 0, jobIds = []) {
    const query = {
      page
    };
    if (jobIds.length > 0) {
      // Only getting the first 10, to prevent URL overflows
      query.ids = jobIds.slice(0, 10).join(',');
    }
    const jobQueueEntries = await this.http.get(`${_constants.API_LIST_URL}/list`, {
      query,
      asSystemRequest: true
    });
    return jobQueueEntries.map(report => new _job.Job(report));
  }
  async total() {
    return await this.http.get(`${_constants.API_LIST_URL}/count`, {
      asSystemRequest: true
    });
  }
  async getError(jobId) {
    var _job$warnings;
    const job = await this.getInfo(jobId);
    if ((_job$warnings = job.warnings) !== null && _job$warnings !== void 0 && _job$warnings[0]) {
      // the error message of a failed report is a singular string in the warnings array
      return job.warnings[0];
    }
    return _i18n.i18n.translate('xpack.reporting.apiClient.unknownError', {
      defaultMessage: `Report job {job} failed. Error unknown.`,
      values: {
        job: jobId
      }
    });
  }
  async getInfo(jobId) {
    const report = await this.http.get(`${_constants.API_LIST_URL}/info/${jobId}`, {
      asSystemRequest: true
    });
    return new _job.Job(report);
  }
  async findForJobIds(jobIds) {
    const reports = await this.http.fetch(`${_constants.API_LIST_URL}/list`, {
      query: {
        page: 0,
        ids: jobIds.join(',')
      },
      method: 'GET'
    });
    return reports.map(report => new _job.Job(report));
  }
  getReportingJobPath(exportType, jobParams) {
    const params = (0, _queryString.stringify)({
      jobParams: _rison.default.encode(jobParams)
    });
    return `${this.http.basePath.prepend(_constants.API_BASE_GENERATE)}/${exportType}?${params}`;
  }
  async createReportingJob(exportType, jobParams) {
    const jobParamsRison = _rison.default.encode(jobParams);
    const resp = await this.http.post(`${_constants.API_BASE_GENERATE}/${exportType}`, {
      method: 'POST',
      body: JSON.stringify({
        jobParams: jobParamsRison
      })
    });
    (0, _job_completion_notifications.add)(resp.job.id);
    return new _job.Job(resp.job);
  }
  async createImmediateReport(baseParams) {
    const {
      objectType: _objectType,
      ...params
    } = baseParams; // objectType is not needed for immediate download api
    return this.http.post(`${_constants.API_GENERATE_IMMEDIATE}`, {
      asResponse: true,
      body: JSON.stringify(params)
    });
  }
  getDecoratedJobParams(baseParams) {
    // If the TZ is set to the default "Browser", it will not be useful for
    // server-side export. We need to derive the timezone and pass it as a param
    // to the export API.
    const browserTimezone = this.uiSettings.get('dateFormat:tz') === 'Browser' ? _moment.default.tz.guess() : this.uiSettings.get('dateFormat:tz');
    return {
      browserTimezone,
      version: this.kibanaVersion,
      ...baseParams
    };
  }
  verifyBrowser() {
    return this.http.post(`${_constants.API_BASE_URL}/diagnose/browser`, {
      asSystemRequest: true
    });
  }
  verifyScreenCapture() {
    return this.http.post(`${_constants.API_BASE_URL}/diagnose/screenshot`, {
      asSystemRequest: true
    });
  }
  migrateReportingIndicesIlmPolicy() {
    return this.http.put(`${_constants.API_MIGRATE_ILM_POLICY_URL}`);
  }
}
exports.ReportingAPIClient = ReportingAPIClient;