"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerSetupRoute = registerSetupRoute;
var _async = require("async");
var _common = require("../../common");
var _get_setup_instructions = require("../lib/setup/get_setup_instructions");
var _steps = require("../lib/setup/steps");
var _handle_route_error_handler = require("../utils/handle_route_error_handler");
var _has_profiling_data = require("../lib/setup/has_profiling_data");
var _compat = require("./compat");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function checkSteps({
  steps,
  logger
}) {
  return Promise.all(steps.map(async step => {
    try {
      return {
        name: step.name,
        completed: await step.hasCompleted()
      };
    } catch (error) {
      logger.error(error);
      return {
        name: step.name,
        completed: false,
        error: error.toString()
      };
    }
  }));
}
function registerSetupRoute({
  router,
  logger,
  services: {
    createProfilingEsClient
  },
  dependencies
}) {
  const paths = (0, _common.getRoutePaths)();
  // Check if ES resources needed for Universal Profiling to work exist
  router.get({
    path: paths.HasSetupESResources,
    validate: false
  }, async (context, request, response) => {
    try {
      const esClient = await (0, _compat.getClient)(context);
      logger.debug('checking if profiling ES configurations are installed');
      const core = await context.core;
      const steps = (0, _steps.getProfilingSetupSteps)({
        client: createProfilingEsClient({
          esClient,
          request,
          useDefaultAuth: true
        }),
        logger,
        packagePolicyClient: dependencies.start.fleet.packagePolicyService,
        soClient: core.savedObjects.client,
        spaceId: dependencies.setup.spaces.spacesService.getSpaceId(request),
        isCloudEnabled: dependencies.setup.cloud.isCloudEnabled
      });
      const hasDataPromise = (0, _has_profiling_data.hasProfilingData)({
        client: createProfilingEsClient({
          esClient,
          request
        })
      });
      const stepCompletionResultsPromises = checkSteps({
        steps,
        logger
      });
      const hasData = await hasDataPromise;
      if (hasData) {
        return response.ok({
          body: {
            has_data: true,
            has_setup: true,
            steps: []
          }
        });
      }
      const stepCompletionResults = await stepCompletionResultsPromises;

      // Reply to clients if we have already created all 12 events template indices.
      // This is kind of simplistic but can be a good first step to ensure
      // Profiling resources will be created.
      return response.ok({
        body: {
          has_setup: stepCompletionResults.every(step => step.completed),
          has_data: false,
          steps: stepCompletionResults
        }
      });
    } catch (error) {
      return (0, _handle_route_error_handler.handleRouteHandlerError)({
        error,
        logger,
        response
      });
    }
  });
  // Configure ES resources needed by Universal Profiling using the mappings
  router.post({
    path: paths.HasSetupESResources,
    validate: {}
  }, async (context, request, response) => {
    try {
      const esClient = await (0, _compat.getClient)(context);
      logger.info('Applying initial setup of Elasticsearch resources');
      const steps = (0, _steps.getProfilingSetupSteps)({
        client: createProfilingEsClient({
          esClient,
          request,
          useDefaultAuth: true
        }),
        logger,
        packagePolicyClient: dependencies.start.fleet.packagePolicyService,
        soClient: (await context.core).savedObjects.client,
        spaceId: dependencies.setup.spaces.spacesService.getSpaceId(request),
        isCloudEnabled: dependencies.setup.cloud.isCloudEnabled
      });
      await (0, _async.eachSeries)(steps, (step, cb) => {
        logger.debug(`Executing step ${step.name}`);
        step.init().then(() => cb()).catch(cb);
      });
      const checkedSteps = await checkSteps({
        steps,
        logger
      });
      if (checkedSteps.every(step => step.completed)) {
        return response.ok();
      }
      return response.custom({
        statusCode: 500,
        body: {
          message: `Failed to complete all steps`,
          steps: checkedSteps
        }
      });
    } catch (error) {
      return (0, _handle_route_error_handler.handleRouteHandlerError)({
        error,
        logger,
        response
      });
    }
  });
  // Show users the instructions on how to setup Universal Profiling agents
  router.get({
    path: paths.SetupDataCollectionInstructions,
    validate: false
  }, async (context, request, response) => {
    try {
      const setupInstructions = await (0, _get_setup_instructions.getSetupInstructions)({
        packagePolicyClient: dependencies.start.fleet.packagePolicyService,
        soClient: (await context.core).savedObjects.client
      });
      return response.ok({
        body: setupInstructions
      });
    } catch (error) {
      return (0, _handle_route_error_handler.handleRouteHandlerError)({
        error,
        logger,
        response
      });
    }
  });
}