"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TopNFunctionsTable = TopNFunctionsTable;
var _eui = require("@elastic/eui");
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _react = _interopRequireWildcard(require("react"));
var _functions = require("../../../common/functions");
var _profiling = require("../../../common/profiling");
var _calculate_impact_estimates = require("../../utils/calculate_impact_estimates");
var _as_cost = require("../../utils/formatters/as_cost");
var _as_weight = require("../../utils/formatters/as_weight");
var _frame_information_tooltip = require("../frame_information_window/frame_information_tooltip");
var _cpu_label_with_hint = require("../shared/cpu_label_with_hint");
var _stack_frame_summary = require("../stack_frame_summary");
var _get_label = require("./get_label");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function TotalSamplesStat({
  totalSamples,
  newSamples
}) {
  const value = totalSamples.toLocaleString();
  const sampleHeader = _i18n.i18n.translate('xpack.profiling.functionsView.totalSampleCountLabel', {
    defaultMessage: ' Total sample estimate: '
  });
  if (newSamples === undefined || newSamples === 0) {
    return /*#__PURE__*/_react.default.createElement(_eui.EuiStat, {
      title: /*#__PURE__*/_react.default.createElement(_eui.EuiText, {
        style: {
          fontWeight: 'bold'
        }
      }, value),
      description: sampleHeader
    });
  }
  const diffSamples = totalSamples - newSamples;
  const percentDelta = diffSamples / (totalSamples - diffSamples) * 100;
  return /*#__PURE__*/_react.default.createElement(_eui.EuiStat, {
    title: /*#__PURE__*/_react.default.createElement(_eui.EuiText, {
      style: {
        fontWeight: 'bold'
      }
    }, value, /*#__PURE__*/_react.default.createElement(_get_label.GetLabel, {
      value: percentDelta,
      prepend: "(",
      append: ")"
    })),
    description: sampleHeader
  });
}
function SampleStat({
  samples,
  diffSamples,
  totalSamples
}) {
  const samplesLabel = `${samples.toLocaleString()}`;
  if (diffSamples === undefined || diffSamples === 0 || totalSamples === 0) {
    return /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, samplesLabel);
  }
  const percentDelta = diffSamples / (samples - diffSamples) * 100;
  const totalPercentDelta = diffSamples / totalSamples * 100;
  return /*#__PURE__*/_react.default.createElement(_eui.EuiFlexGroup, {
    direction: "column",
    gutterSize: "none"
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, null, samplesLabel), /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, null, /*#__PURE__*/_react.default.createElement(_get_label.GetLabel, {
    value: percentDelta,
    append: " rel"
  })), /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, null, /*#__PURE__*/_react.default.createElement(_get_label.GetLabel, {
    value: totalPercentDelta,
    append: " abs"
  })));
}
function CPUStat({
  cpu,
  diffCPU
}) {
  const cpuLabel = `${cpu.toFixed(2)}%`;
  if (diffCPU === undefined || diffCPU === 0) {
    return /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, cpuLabel);
  }
  return /*#__PURE__*/_react.default.createElement(_eui.EuiFlexGroup, {
    direction: "column",
    gutterSize: "none"
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, null, cpuLabel), /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, null, /*#__PURE__*/_react.default.createElement(_get_label.GetLabel, {
    value: diffCPU,
    prepend: "(",
    append: ")"
  })));
}
function TopNFunctionsTable({
  sortDirection,
  sortField,
  onSortChange,
  topNFunctions,
  comparisonTopNFunctions,
  totalSeconds,
  isDifferentialView
}) {
  const [selectedRow, setSelectedRow] = (0, _react.useState)();
  const totalCount = (0, _react.useMemo)(() => {
    if (!topNFunctions || !topNFunctions.TotalCount) {
      return 0;
    }
    return topNFunctions.TotalCount;
  }, [topNFunctions]);
  const rows = (0, _react.useMemo)(() => {
    if (!topNFunctions || !topNFunctions.TotalCount || topNFunctions.TotalCount === 0) {
      return [];
    }
    const comparisonDataById = comparisonTopNFunctions ? (0, _lodash.keyBy)(comparisonTopNFunctions.TopN, 'Id') : {};
    return topNFunctions.TopN.filter(topN => topN.CountExclusive > 0).map((topN, i) => {
      const comparisonRow = comparisonDataById === null || comparisonDataById === void 0 ? void 0 : comparisonDataById[topN.Id];
      const inclusiveCPU = topN.CountInclusive / topNFunctions.TotalCount * 100;
      const exclusiveCPU = topN.CountExclusive / topNFunctions.TotalCount * 100;
      const totalSamples = topN.CountExclusive;
      const impactEstimates = totalSeconds > 0 ? (0, _calculate_impact_estimates.calculateImpactEstimates)({
        countExclusive: exclusiveCPU,
        countInclusive: inclusiveCPU,
        totalSamples,
        totalSeconds
      }) : undefined;
      const diff = comparisonTopNFunctions && comparisonRow ? {
        rank: topN.Rank - comparisonRow.Rank,
        samples: topN.CountExclusive - comparisonRow.CountExclusive,
        exclusiveCPU: exclusiveCPU - comparisonRow.CountExclusive / comparisonTopNFunctions.TotalCount * 100,
        inclusiveCPU: inclusiveCPU - comparisonRow.CountInclusive / comparisonTopNFunctions.TotalCount * 100
      } : undefined;
      return {
        rank: topN.Rank,
        frame: topN.Frame,
        samples: topN.CountExclusive,
        exclusiveCPU,
        inclusiveCPU,
        impactEstimates,
        diff
      };
    });
  }, [topNFunctions, comparisonTopNFunctions, totalSeconds]);
  const theme = (0, _eui.useEuiTheme)();
  const columns = [{
    field: _functions.TopNFunctionSortField.Rank,
    name: _i18n.i18n.translate('xpack.profiling.functionsView.rankColumnLabel', {
      defaultMessage: 'Rank'
    }),
    render: (_, {
      rank
    }) => {
      return /*#__PURE__*/_react.default.createElement(_eui.EuiText, {
        style: {
          whiteSpace: 'nowrap',
          fontSize: 'inherit'
        }
      }, rank);
    },
    align: 'right'
  }, {
    field: _functions.TopNFunctionSortField.Frame,
    name: _i18n.i18n.translate('xpack.profiling.functionsView.functionColumnLabel', {
      defaultMessage: 'Function'
    }),
    render: (_, {
      frame
    }) => /*#__PURE__*/_react.default.createElement(_stack_frame_summary.StackFrameSummary, {
      frame: frame
    }),
    width: '50%'
  }, {
    field: _functions.TopNFunctionSortField.Samples,
    name: _i18n.i18n.translate('xpack.profiling.functionsView.samplesColumnLabel', {
      defaultMessage: 'Samples (estd.)'
    }),
    render: (_, {
      samples,
      diff
    }) => {
      return /*#__PURE__*/_react.default.createElement(SampleStat, {
        samples: samples,
        diffSamples: diff === null || diff === void 0 ? void 0 : diff.samples,
        totalSamples: totalCount
      });
    },
    align: 'right'
  }, {
    field: _functions.TopNFunctionSortField.ExclusiveCPU,
    name: /*#__PURE__*/_react.default.createElement(_cpu_label_with_hint.CPULabelWithHint, {
      type: "self",
      labelSize: "xs",
      labelStyle: {
        fontWeight: 600
      },
      iconSize: "s"
    }),
    render: (_, {
      exclusiveCPU,
      diff
    }) => {
      return /*#__PURE__*/_react.default.createElement(CPUStat, {
        cpu: exclusiveCPU,
        diffCPU: diff === null || diff === void 0 ? void 0 : diff.exclusiveCPU
      });
    },
    align: 'right'
  }, {
    field: _functions.TopNFunctionSortField.InclusiveCPU,
    name: /*#__PURE__*/_react.default.createElement(_cpu_label_with_hint.CPULabelWithHint, {
      type: "total",
      labelSize: "xs",
      labelStyle: {
        fontWeight: 600
      },
      iconSize: "s"
    }),
    render: (_, {
      inclusiveCPU,
      diff
    }) => {
      return /*#__PURE__*/_react.default.createElement(CPUStat, {
        cpu: inclusiveCPU,
        diffCPU: diff === null || diff === void 0 ? void 0 : diff.inclusiveCPU
      });
    },
    align: 'right'
  }];
  if (comparisonTopNFunctions) {
    columns.push({
      field: _functions.TopNFunctionSortField.Diff,
      name: _i18n.i18n.translate('xpack.profiling.functionsView.diffColumnLabel', {
        defaultMessage: 'Diff'
      }),
      align: 'right',
      render: (_, {
        diff
      }) => {
        if (!diff) {
          return /*#__PURE__*/_react.default.createElement(_eui.EuiText, {
            size: "xs",
            color: theme.euiTheme.colors.primaryText
          }, _i18n.i18n.translate('xpack.profiling.functionsView.newLabel', {
            defaultMessage: 'New'
          }));
        }
        if (diff.rank === 0) {
          return null;
        }
        const color = diff.rank > 0 ? 'success' : 'danger';
        const icon = diff.rank > 0 ? 'sortDown' : 'sortUp';
        return /*#__PURE__*/_react.default.createElement(_eui.EuiBadge, {
          color: color,
          iconType: icon,
          iconSide: "right",
          style: {
            minWidth: '100%',
            textAlign: 'right'
          }
        }, diff.rank);
      }
    });
  }
  if (!isDifferentialView) {
    columns.push({
      field: 'annualized_co2',
      name: _i18n.i18n.translate('xpack.profiling.functionsView.annualizedCo2', {
        defaultMessage: 'Annualized CO2'
      }),
      render: (_, {
        impactEstimates
      }) => {
        if (impactEstimates !== null && impactEstimates !== void 0 && impactEstimates.annualizedCo2) {
          return /*#__PURE__*/_react.default.createElement("div", null, (0, _as_weight.asWeight)(impactEstimates.annualizedCo2));
        }
      },
      align: 'right'
    }, {
      field: 'annualized_dollar_cost',
      name: _i18n.i18n.translate('xpack.profiling.functionsView.annualizedDollarCost', {
        defaultMessage: `Annualized dollar cost`
      }),
      render: (_, {
        impactEstimates
      }) => {
        if (impactEstimates !== null && impactEstimates !== void 0 && impactEstimates.annualizedDollarCost) {
          return /*#__PURE__*/_react.default.createElement("div", null, (0, _as_cost.asCost)(impactEstimates.annualizedDollarCost));
        }
      },
      align: 'right'
    }, {
      name: 'Actions',
      actions: [{
        name: 'show_more_information',
        description: _i18n.i18n.translate('xpack.profiling.functionsView.showMoreButton', {
          defaultMessage: `Show more information`
        }),
        icon: 'inspect',
        color: 'primary',
        type: 'icon',
        onClick: setSelectedRow
      }]
    });
  }
  const sortedRows = (0, _lodash.orderBy)(rows, row => {
    return sortField === _functions.TopNFunctionSortField.Frame ? (0, _profiling.getCalleeFunction)(row.frame).toLowerCase() : row[sortField];
  }, [sortDirection]).slice(0, 100);
  return /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement(TotalSamplesStat, {
    totalSamples: totalCount,
    newSamples: comparisonTopNFunctions === null || comparisonTopNFunctions === void 0 ? void 0 : comparisonTopNFunctions.TotalCount
  }), /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
    size: "s"
  }), /*#__PURE__*/_react.default.createElement(_eui.EuiHorizontalRule, {
    margin: "none",
    style: {
      height: 2
    }
  }), /*#__PURE__*/_react.default.createElement(_eui.EuiBasicTable, {
    items: sortedRows,
    columns: columns,
    tableLayout: "auto",
    onChange: criteria => {
      onSortChange({
        sortDirection: criteria.sort.direction,
        sortField: criteria.sort.field
      });
    },
    sorting: {
      enableAllColumns: true,
      sort: {
        direction: sortDirection,
        field: sortField
      }
    }
  }), selectedRow && /*#__PURE__*/_react.default.createElement(_frame_information_tooltip.FrameInformationTooltip, {
    onClose: () => {
      setSelectedRow(undefined);
    },
    frame: {
      addressOrLine: selectedRow.frame.AddressOrLine,
      countExclusive: selectedRow.exclusiveCPU,
      countInclusive: selectedRow.inclusiveCPU,
      exeFileName: selectedRow.frame.ExeFileName,
      fileID: selectedRow.frame.FileID,
      frameType: selectedRow.frame.FrameType,
      functionName: selectedRow.frame.FunctionName,
      sourceFileName: selectedRow.frame.SourceFilename,
      sourceLine: selectedRow.frame.SourceLine
    },
    totalSeconds: totalSeconds !== null && totalSeconds !== void 0 ? totalSeconds : 0,
    totalSamples: selectedRow.samples
  }));
}