"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SortField = exports.SortDirection = exports.KibanaSavedObjectsSLORepository = void 0;
var t = _interopRequireWildcard(require("io-ts"));
var _Either = require("fp-ts/lib/Either");
var _pipeable = require("fp-ts/lib/pipeable");
var _coreSavedObjectsServer = require("@kbn/core-saved-objects-server");
var _sloSchema = require("@kbn/slo-schema");
var _saved_objects = require("../../saved_objects");
var _errors = require("../../errors");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SortDirection = {
  Asc: 'Asc',
  Desc: 'Desc'
};
exports.SortDirection = SortDirection;
const SortField = {
  CreationTime: 'CreationTime',
  IndicatorType: 'IndicatorType'
};
exports.SortField = SortField;
class KibanaSavedObjectsSLORepository {
  constructor(soClient) {
    this.soClient = soClient;
  }
  async save(slo) {
    const savedSLO = await this.soClient.create(_saved_objects.SO_SLO_TYPE, toStoredSLO(slo), {
      id: slo.id,
      overwrite: true
    });
    return toSLO(savedSLO.attributes);
  }
  async findById(id) {
    try {
      const slo = await this.soClient.get(_saved_objects.SO_SLO_TYPE, id);
      return toSLO(slo.attributes);
    } catch (err) {
      if (_coreSavedObjectsServer.SavedObjectsErrorHelpers.isNotFoundError(err)) {
        throw new _errors.SLONotFound(`SLO [${id}] not found`);
      }
      throw err;
    }
  }
  async deleteById(id) {
    try {
      await this.soClient.delete(_saved_objects.SO_SLO_TYPE, id);
    } catch (err) {
      if (_coreSavedObjectsServer.SavedObjectsErrorHelpers.isNotFoundError(err)) {
        throw new _errors.SLONotFound(`SLO [${id}] not found`);
      }
      throw err;
    }
  }
  async find(criteria, sort, pagination) {
    const {
      search,
      searchFields
    } = buildSearch(criteria);
    const filterKuery = buildFilterKuery(criteria);
    const {
      sortField,
      sortOrder
    } = buildSortQuery(sort);
    const response = await this.soClient.find({
      type: _saved_objects.SO_SLO_TYPE,
      page: pagination.page,
      perPage: pagination.perPage,
      search,
      searchFields,
      filter: filterKuery,
      sortField,
      sortOrder
    });
    return {
      total: response.total,
      page: response.page,
      perPage: response.per_page,
      results: response.saved_objects.map(slo => toSLO(slo.attributes))
    };
  }
  async findAllByIds(ids) {
    if (ids.length === 0) return [];
    try {
      const response = await this.soClient.find({
        type: _saved_objects.SO_SLO_TYPE,
        page: 1,
        perPage: ids.length,
        filter: `slo.attributes.id:(${ids.join(' or ')})`
      });
      return response.saved_objects.map(slo => toSLO(slo.attributes));
    } catch (err) {
      if (_coreSavedObjectsServer.SavedObjectsErrorHelpers.isNotFoundError(err)) {
        throw new _errors.SLONotFound(`SLOs [${ids.join(',')}] not found`);
      }
      throw err;
    }
  }
}
exports.KibanaSavedObjectsSLORepository = KibanaSavedObjectsSLORepository;
function buildSearch(criteria) {
  if (!criteria.name) {
    return {
      search: undefined,
      searchFields: undefined
    };
  }
  return {
    search: addWildcardsIfAbsent(criteria.name),
    searchFields: ['name']
  };
}
function buildFilterKuery(criteria) {
  const filters = [];
  if (!!criteria.indicatorTypes) {
    const indicatorTypesFilter = criteria.indicatorTypes.map(indicatorType => `slo.attributes.indicator.type: ${indicatorType}`);
    filters.push(`(${indicatorTypesFilter.join(' or ')})`);
  }
  return filters.length > 0 ? filters.join(' and ') : undefined;
}
function buildSortQuery(sort) {
  let sortField;
  switch (sort.field) {
    case SortField.IndicatorType:
      sortField = 'indicator.type';
      break;
    case SortField.CreationTime:
    default:
      sortField = 'created_at';
      break;
  }
  return {
    sortField,
    sortOrder: sort.direction === SortDirection.Desc ? 'desc' : 'asc'
  };
}
function toStoredSLO(slo) {
  return _sloSchema.sloSchema.encode(slo);
}
function toSLO(storedSLO) {
  return (0, _pipeable.pipe)(_sloSchema.sloSchema.decode(storedSLO), (0, _Either.fold)(() => {
    throw new Error('Invalid Stored SLO');
  }, t.identity));
}
const WILDCARD_CHAR = '*';
function addWildcardsIfAbsent(value) {
  let updatedValue = value;
  if (updatedValue.substring(0, 1) !== WILDCARD_CHAR) {
    updatedValue = `${WILDCARD_CHAR}${updatedValue}`;
  }
  if (value.substring(value.length - 1) !== WILDCARD_CHAR) {
    updatedValue = `${updatedValue}${WILDCARD_CHAR}`;
  }
  return updatedValue;
}