"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.slosRouteRepository = void 0;
var _boom = require("@hapi/boom");
var _sloSchema = require("@kbn/slo-schema");
var _slo = require("../../services/slo");
var _transform_generators = require("../../services/slo/transform_generators");
var _create_observability_server_route = require("../create_observability_server_route");
var _historical_summary_client = require("../../services/slo/historical_summary_client");
var _fetch_historical_summary = require("../../services/slo/fetch_historical_summary");
var _manage_slo = require("../../services/slo/manage_slo");
var _get_diagnosis = require("../../services/slo/get_diagnosis");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const transformGenerators = {
  'sli.apm.transactionDuration': new _transform_generators.ApmTransactionDurationTransformGenerator(),
  'sli.apm.transactionErrorRate': new _transform_generators.ApmTransactionErrorRateTransformGenerator(),
  'sli.kql.custom': new _transform_generators.KQLCustomTransformGenerator()
};
const isLicenseAtLeastPlatinum = async context => {
  const licensing = await context.licensing;
  return licensing.license.hasAtLeast('platinum');
};
const createSLORoute = (0, _create_observability_server_route.createObservabilityServerRoute)({
  endpoint: 'POST /api/observability/slos',
  options: {
    tags: ['access:slo_write']
  },
  params: _sloSchema.createSLOParamsSchema,
  handler: async ({
    context,
    params,
    logger
  }) => {
    const hasCorrectLicense = await isLicenseAtLeastPlatinum(context);
    if (!hasCorrectLicense) {
      throw (0, _boom.badRequest)('Platinum license or higher is needed to make use of this feature.');
    }
    const esClient = (await context.core).elasticsearch.client.asCurrentUser;
    const soClient = (await context.core).savedObjects.client;
    const resourceInstaller = new _slo.DefaultResourceInstaller(esClient, logger);
    const repository = new _slo.KibanaSavedObjectsSLORepository(soClient);
    const transformManager = new _slo.DefaultTransformManager(transformGenerators, esClient, logger);
    const createSLO = new _slo.CreateSLO(resourceInstaller, repository, transformManager);
    const response = await createSLO.execute(params.body);
    return response;
  }
});
const updateSLORoute = (0, _create_observability_server_route.createObservabilityServerRoute)({
  endpoint: 'PUT /api/observability/slos/{id}',
  options: {
    tags: ['access:slo_write']
  },
  params: _sloSchema.updateSLOParamsSchema,
  handler: async ({
    context,
    params,
    logger
  }) => {
    const hasCorrectLicense = await isLicenseAtLeastPlatinum(context);
    if (!hasCorrectLicense) {
      throw (0, _boom.badRequest)('Platinum license or higher is needed to make use of this feature.');
    }
    const esClient = (await context.core).elasticsearch.client.asCurrentUser;
    const soClient = (await context.core).savedObjects.client;
    const repository = new _slo.KibanaSavedObjectsSLORepository(soClient);
    const transformManager = new _slo.DefaultTransformManager(transformGenerators, esClient, logger);
    const updateSLO = new _slo.UpdateSLO(repository, transformManager, esClient);
    const response = await updateSLO.execute(params.path.id, params.body);
    return response;
  }
});
const deleteSLORoute = (0, _create_observability_server_route.createObservabilityServerRoute)({
  endpoint: 'DELETE /api/observability/slos/{id}',
  options: {
    tags: ['access:slo_write']
  },
  params: _sloSchema.deleteSLOParamsSchema,
  handler: async ({
    request,
    context,
    params,
    logger,
    dependencies: {
      getRulesClientWithRequest
    }
  }) => {
    const hasCorrectLicense = await isLicenseAtLeastPlatinum(context);
    if (!hasCorrectLicense) {
      throw (0, _boom.badRequest)('Platinum license or higher is needed to make use of this feature.');
    }
    const esClient = (await context.core).elasticsearch.client.asCurrentUser;
    const soClient = (await context.core).savedObjects.client;
    const rulesClient = getRulesClientWithRequest(request);
    const repository = new _slo.KibanaSavedObjectsSLORepository(soClient);
    const transformManager = new _slo.DefaultTransformManager(transformGenerators, esClient, logger);
    const deleteSLO = new _slo.DeleteSLO(repository, transformManager, esClient, rulesClient);
    await deleteSLO.execute(params.path.id);
  }
});
const getSLORoute = (0, _create_observability_server_route.createObservabilityServerRoute)({
  endpoint: 'GET /api/observability/slos/{id}',
  options: {
    tags: ['access:slo_read']
  },
  params: _sloSchema.getSLOParamsSchema,
  handler: async ({
    context,
    params
  }) => {
    const hasCorrectLicense = await isLicenseAtLeastPlatinum(context);
    if (!hasCorrectLicense) {
      throw (0, _boom.badRequest)('Platinum license or higher is needed to make use of this feature.');
    }
    const soClient = (await context.core).savedObjects.client;
    const esClient = (await context.core).elasticsearch.client.asCurrentUser;
    const repository = new _slo.KibanaSavedObjectsSLORepository(soClient);
    const summaryClient = new _slo.DefaultSummaryClient(esClient);
    const getSLO = new _slo.GetSLO(repository, summaryClient);
    const response = await getSLO.execute(params.path.id);
    return response;
  }
});
const enableSLORoute = (0, _create_observability_server_route.createObservabilityServerRoute)({
  endpoint: 'POST /api/observability/slos/{id}/enable',
  options: {
    tags: ['access:slo_write']
  },
  params: _sloSchema.manageSLOParamsSchema,
  handler: async ({
    context,
    params,
    logger
  }) => {
    const hasCorrectLicense = await isLicenseAtLeastPlatinum(context);
    if (!hasCorrectLicense) {
      throw (0, _boom.badRequest)('Platinum license or higher is needed to make use of this feature.');
    }
    const soClient = (await context.core).savedObjects.client;
    const esClient = (await context.core).elasticsearch.client.asCurrentUser;
    const repository = new _slo.KibanaSavedObjectsSLORepository(soClient);
    const transformManager = new _slo.DefaultTransformManager(transformGenerators, esClient, logger);
    const manageSLO = new _manage_slo.ManageSLO(repository, transformManager);
    const response = await manageSLO.enable(params.path.id);
    return response;
  }
});
const disableSLORoute = (0, _create_observability_server_route.createObservabilityServerRoute)({
  endpoint: 'POST /api/observability/slos/{id}/disable',
  options: {
    tags: ['access:slo_write']
  },
  params: _sloSchema.manageSLOParamsSchema,
  handler: async ({
    context,
    params,
    logger
  }) => {
    const hasCorrectLicense = await isLicenseAtLeastPlatinum(context);
    if (!hasCorrectLicense) {
      throw (0, _boom.badRequest)('Platinum license or higher is needed to make use of this feature.');
    }
    const soClient = (await context.core).savedObjects.client;
    const esClient = (await context.core).elasticsearch.client.asCurrentUser;
    const repository = new _slo.KibanaSavedObjectsSLORepository(soClient);
    const transformManager = new _slo.DefaultTransformManager(transformGenerators, esClient, logger);
    const manageSLO = new _manage_slo.ManageSLO(repository, transformManager);
    const response = await manageSLO.disable(params.path.id);
    return response;
  }
});
const findSLORoute = (0, _create_observability_server_route.createObservabilityServerRoute)({
  endpoint: 'GET /api/observability/slos',
  options: {
    tags: ['access:slo_read']
  },
  params: _sloSchema.findSLOParamsSchema,
  handler: async ({
    context,
    params
  }) => {
    var _params$query;
    const hasCorrectLicense = await isLicenseAtLeastPlatinum(context);
    if (!hasCorrectLicense) {
      throw (0, _boom.badRequest)('Platinum license or higher is needed to make use of this feature.');
    }
    const soClient = (await context.core).savedObjects.client;
    const esClient = (await context.core).elasticsearch.client.asCurrentUser;
    const repository = new _slo.KibanaSavedObjectsSLORepository(soClient);
    const summaryClient = new _slo.DefaultSummaryClient(esClient);
    const findSLO = new _slo.FindSLO(repository, summaryClient);
    const response = await findSLO.execute((_params$query = params === null || params === void 0 ? void 0 : params.query) !== null && _params$query !== void 0 ? _params$query : {});
    return response;
  }
});
const fetchHistoricalSummary = (0, _create_observability_server_route.createObservabilityServerRoute)({
  endpoint: 'POST /internal/observability/slos/_historical_summary',
  options: {
    tags: ['access:slo_read']
  },
  params: _sloSchema.fetchHistoricalSummaryParamsSchema,
  handler: async ({
    context,
    params
  }) => {
    const hasCorrectLicense = await isLicenseAtLeastPlatinum(context);
    if (!hasCorrectLicense) {
      throw (0, _boom.badRequest)('Platinum license or higher is needed to make use of this feature.');
    }
    const soClient = (await context.core).savedObjects.client;
    const esClient = (await context.core).elasticsearch.client.asCurrentUser;
    const repository = new _slo.KibanaSavedObjectsSLORepository(soClient);
    const historicalSummaryClient = new _historical_summary_client.DefaultHistoricalSummaryClient(esClient);
    const fetchSummaryData = new _fetch_historical_summary.FetchHistoricalSummary(repository, historicalSummaryClient);
    const response = await fetchSummaryData.execute(params.body);
    return response;
  }
});
const getDiagnosisRoute = (0, _create_observability_server_route.createObservabilityServerRoute)({
  endpoint: 'GET /internal/observability/slos/_diagnosis',
  options: {
    tags: []
  },
  params: undefined,
  handler: async ({
    context
  }) => {
    const esClient = (await context.core).elasticsearch.client.asCurrentUser;
    const licensing = await context.licensing;
    try {
      const response = await (0, _get_diagnosis.getGlobalDiagnosis)(esClient, licensing);
      return response;
    } catch (error) {
      if (error.cause.statusCode === 403) {
        throw (0, _boom.forbidden)('Insufficient Elasticsearch cluster permissions to access feature.');
      }
      throw (0, _boom.failedDependency)(error);
    }
  }
});
const getSloDiagnosisRoute = (0, _create_observability_server_route.createObservabilityServerRoute)({
  endpoint: 'GET /internal/observability/slos/{id}/_diagnosis',
  options: {
    tags: []
  },
  params: _sloSchema.getSLODiagnosisParamsSchema,
  handler: async ({
    context,
    params
  }) => {
    const esClient = (await context.core).elasticsearch.client.asCurrentUser;
    const soClient = (await context.core).savedObjects.client;
    return (0, _get_diagnosis.getSloDiagnosis)(params.path.id, {
      esClient,
      soClient
    });
  }
});
const slosRouteRepository = {
  ...createSLORoute,
  ...deleteSLORoute,
  ...disableSLORoute,
  ...enableSLORoute,
  ...fetchHistoricalSummary,
  ...findSLORoute,
  ...getSLORoute,
  ...updateSLORoute,
  ...getDiagnosisRoute,
  ...getSloDiagnosisRoute
};
exports.slosRouteRepository = slosRouteRepository;