"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getRuleExecutor = exports.ALERT_ACTION = void 0;
var _i18n = require("@kbn/i18n");
var _numeral = _interopRequireDefault(require("@elastic/numeral"));
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _server = require("@kbn/spaces-plugin/server");
var _infra_metrics = require("../../../../common/field_names/infra_metrics");
var _models = require("../../../domain/models");
var _slo = require("../../../services/slo");
var _services = require("../../../domain/services");
var _types = require("./types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SHORT_WINDOW = 'SHORT_WINDOW';
const LONG_WINDOW = 'LONG_WINDOW';
const getRuleExecutor = ({
  basePath
}) => async function executor({
  services,
  params,
  startedAt,
  spaceId
}) {
  const {
    alertWithLifecycle,
    savedObjectsClient: soClient,
    scopedClusterClient: esClient,
    alertFactory
  } = services;
  const sloRepository = new _slo.KibanaSavedObjectsSLORepository(soClient);
  const summaryClient = new _slo.DefaultSLIClient(esClient.asCurrentUser);
  const slo = await sloRepository.findById(params.sloId);
  if (!slo.enabled) {
    return {
      state: {}
    };
  }
  const longWindowDuration = new _models.Duration(params.longWindow.value, (0, _models.toDurationUnit)(params.longWindow.unit));
  const shortWindowDuration = new _models.Duration(params.shortWindow.value, (0, _models.toDurationUnit)(params.shortWindow.unit));
  const sliData = await summaryClient.fetchSLIDataFrom(slo, [{
    name: LONG_WINDOW,
    duration: longWindowDuration.add(slo.settings.syncDelay)
  }, {
    name: SHORT_WINDOW,
    duration: shortWindowDuration.add(slo.settings.syncDelay)
  }]);
  const longWindowBurnRate = (0, _services.computeBurnRate)(slo, sliData[LONG_WINDOW]);
  const shortWindowBurnRate = (0, _services.computeBurnRate)(slo, sliData[SHORT_WINDOW]);
  const shouldAlert = longWindowBurnRate >= params.burnRateThreshold && shortWindowBurnRate >= params.burnRateThreshold;
  const viewInAppUrl = (0, _server.addSpaceIdToPath)(basePath.publicBaseUrl, spaceId, `/app/observability/slos/${slo.id}`);
  if (shouldAlert) {
    const reason = buildReason(longWindowDuration, longWindowBurnRate, shortWindowDuration, shortWindowBurnRate, params);
    const context = {
      longWindow: {
        burnRate: longWindowBurnRate,
        duration: longWindowDuration.format()
      },
      reason,
      shortWindow: {
        burnRate: shortWindowBurnRate,
        duration: shortWindowDuration.format()
      },
      burnRateThreshold: params.burnRateThreshold,
      timestamp: startedAt.toISOString(),
      viewInAppUrl,
      sloId: slo.id,
      sloName: slo.name
    };
    const alert = alertWithLifecycle({
      id: `alert-${slo.id}-${slo.revision}`,
      fields: {
        [_ruleDataUtils.ALERT_REASON]: reason,
        [_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]: params.burnRateThreshold,
        [_ruleDataUtils.ALERT_EVALUATION_VALUE]: Math.min(longWindowBurnRate, shortWindowBurnRate),
        [_infra_metrics.SLO_ID_FIELD]: slo.id,
        [_infra_metrics.SLO_REVISION_FIELD]: slo.revision
      }
    });
    alert.scheduleActions(ALERT_ACTION.id, context);
    alert.replaceState({
      alertState: _types.AlertStates.ALERT
    });
  }
  const {
    getRecoveredAlerts
  } = alertFactory.done();
  const recoveredAlerts = getRecoveredAlerts();
  for (const recoveredAlert of recoveredAlerts) {
    const context = {
      longWindow: {
        burnRate: longWindowBurnRate,
        duration: longWindowDuration.format()
      },
      shortWindow: {
        burnRate: shortWindowBurnRate,
        duration: shortWindowDuration.format()
      },
      burnRateThreshold: params.burnRateThreshold,
      timestamp: startedAt.toISOString(),
      viewInAppUrl,
      sloId: slo.id,
      sloName: slo.name
    };
    recoveredAlert.setContext(context);
  }
  return {
    state: {}
  };
};
exports.getRuleExecutor = getRuleExecutor;
const ALERT_ACTION_ID = 'slo.burnRate.alert';
const ALERT_ACTION = {
  id: ALERT_ACTION_ID,
  name: _i18n.i18n.translate('xpack.observability.slo.alerting.burnRate.alertAction', {
    defaultMessage: 'Alert'
  })
};
exports.ALERT_ACTION = ALERT_ACTION;
function buildReason(longWindowDuration, longWindowBurnRate, shortWindowDuration, shortWindowBurnRate, params) {
  return _i18n.i18n.translate('xpack.observability.slo.alerting.burnRate.reason', {
    defaultMessage: 'The burn rate for the past {longWindowDuration} is {longWindowBurnRate} and for the past {shortWindowDuration} is {shortWindowBurnRate}. Alert when above {burnRateThreshold} for both windows',
    values: {
      longWindowDuration: longWindowDuration.format(),
      longWindowBurnRate: (0, _numeral.default)(longWindowBurnRate).format('0.[00]'),
      shortWindowDuration: shortWindowDuration.format(),
      shortWindowBurnRate: (0, _numeral.default)(shortWindowBurnRate).format('0.[00]'),
      burnRateThreshold: params.burnRateThreshold
    }
  });
}