"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.computeErrorBudget = computeErrorBudget;
exports.computeTotalSlicesFromDateRange = computeTotalSlicesFromDateRange;
exports.toErrorBudget = toErrorBudget;
var _moment = _interopRequireDefault(require("moment"));
var _sloSchema = require("@kbn/slo-schema");
var _models = require("../models");
var _number = require("../../utils/number");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// More details about calculus: https://github.com/elastic/kibana/issues/143980
function computeErrorBudget(slo, sliData) {
  const {
    good,
    total
  } = sliData;
  if (total === 0 || good >= total) {
    const initialErrorBudget = 1 - slo.objective.target;
    return toErrorBudget(initialErrorBudget, 0);
  }
  if (_sloSchema.rollingTimeWindowSchema.is(slo.timeWindow)) {
    return computeForRolling(slo, sliData);
  }
  if (_sloSchema.calendarAlignedTimeWindowSchema.is(slo.timeWindow)) {
    if (_sloSchema.timeslicesBudgetingMethodSchema.is(slo.budgetingMethod)) {
      return computeForCalendarAlignedWithTimeslices(slo, sliData);
    }
    if (_sloSchema.occurrencesBudgetingMethodSchema.is(slo.budgetingMethod)) {
      return computeForCalendarAlignedWithOccurrences(slo, sliData);
    }
  }
  throw new Error('Invalid slo time window');
}
function computeForRolling(slo, sliData) {
  const {
    good,
    total
  } = sliData;
  const initialErrorBudget = 1 - slo.objective.target;
  const consumedErrorBudget = (total - good) / (total * initialErrorBudget);
  return toErrorBudget(initialErrorBudget, consumedErrorBudget);
}
function computeForCalendarAlignedWithOccurrences(slo, sliData) {
  const {
    good,
    total,
    dateRange
  } = sliData;
  const initialErrorBudget = 1 - slo.objective.target;
  const now = (0, _moment.default)();
  const durationCalendarPeriod = (0, _moment.default)(dateRange.to).diff(dateRange.from, 'minutes');
  const durationSinceBeginning = now.isAfter(dateRange.to) ? durationCalendarPeriod : (0, _moment.default)(now).diff(dateRange.from, 'minutes');
  const totalEventsEstimatedAtPeriodEnd = Math.round(total / durationSinceBeginning * durationCalendarPeriod);
  const consumedErrorBudget = (total - good) / (totalEventsEstimatedAtPeriodEnd * initialErrorBudget);
  return toErrorBudget(initialErrorBudget, consumedErrorBudget, true);
}
function computeForCalendarAlignedWithTimeslices(slo, sliData) {
  const {
    good,
    total,
    dateRange
  } = sliData;
  const initialErrorBudget = 1 - slo.objective.target;
  const totalSlices = computeTotalSlicesFromDateRange(dateRange, slo.objective.timesliceWindow);
  const consumedErrorBudget = (total - good) / (totalSlices * initialErrorBudget);
  return toErrorBudget(initialErrorBudget, consumedErrorBudget);
}
function computeTotalSlicesFromDateRange(dateRange, timesliceWindow) {
  const dateRangeDurationInUnit = (0, _moment.default)(dateRange.to).diff(dateRange.from, (0, _models.toMomentUnitOfTime)(timesliceWindow.unit));
  const totalSlices = Math.ceil(dateRangeDurationInUnit / timesliceWindow.value);
  return totalSlices;
}
function toErrorBudget(initial, consumed, isEstimated = false) {
  return {
    initial: (0, _number.toHighPrecision)(initial),
    consumed: (0, _number.toHighPrecision)(consumed),
    remaining: (0, _number.toHighPrecision)(1 - consumed),
    isEstimated
  };
}