"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useFetchSloList = useFetchSloList;
var _react = require("react");
var _reactQuery = require("@tanstack/react-query");
var _kibana_react = require("../../utils/kibana_react");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SHORT_REFETCH_INTERVAL = 1000 * 5; // 5 seconds
const LONG_REFETCH_INTERVAL = 1000 * 60; // 1 minute

function useFetchSloList({
  name = '',
  page = 1,
  sortBy = 'creationTime',
  indicatorTypes = [],
  shouldRefetch
} = {}) {
  const {
    http
  } = (0, _kibana_react.useKibana)().services;
  const queryClient = (0, _reactQuery.useQueryClient)();
  const [stateRefetchInterval, setStateRefetchInterval] = (0, _react.useState)(SHORT_REFETCH_INTERVAL);
  const {
    isInitialLoading,
    isLoading,
    isError,
    isSuccess,
    isRefetching,
    data,
    refetch
  } = (0, _reactQuery.useQuery)({
    queryKey: ['fetchSloList', {
      name,
      page,
      sortBy,
      indicatorTypes
    }],
    queryFn: async ({
      signal
    }) => {
      try {
        const response = await http.get(`/api/observability/slos`, {
          query: {
            ...(page && {
              page
            }),
            ...(name && {
              name
            }),
            ...(sortBy && {
              sortBy
            }),
            ...(indicatorTypes && indicatorTypes.length > 0 && {
              indicatorTypes: indicatorTypes.join(',')
            })
          },
          signal
        });
        return response;
      } catch (error) {
        // ignore error
      }
    },
    keepPreviousData: true,
    refetchOnWindowFocus: false,
    refetchInterval: shouldRefetch ? stateRefetchInterval : undefined,
    staleTime: 1000,
    onSuccess: ({
      results
    }) => {
      if (!shouldRefetch) {
        return;
      }
      if (results.find(slo => slo.summary.status === 'NO_DATA' || !slo.summary)) {
        setStateRefetchInterval(SHORT_REFETCH_INTERVAL);
      } else {
        setStateRefetchInterval(LONG_REFETCH_INTERVAL);
      }
      queryClient.invalidateQueries(['fetchHistoricalSummary'], {
        exact: false
      });
      queryClient.invalidateQueries(['fetchActiveAlerts'], {
        exact: false
      });
      queryClient.invalidateQueries(['fetchRulesForSlo'], {
        exact: false
      });
    }
  });
  return {
    sloList: data,
    isInitialLoading,
    isLoading,
    isRefetching,
    isSuccess,
    isError,
    refetch
  };
}