"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useDeleteSlo = useDeleteSlo;
var _reactQuery = require("@tanstack/react-query");
var _i18n = require("@kbn/i18n");
var _kibana_react = require("../../utils/kibana_react");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function useDeleteSlo() {
  const {
    http,
    notifications: {
      toasts
    }
  } = (0, _kibana_react.useKibana)().services;
  const queryClient = (0, _reactQuery.useQueryClient)();
  const deleteSlo = (0, _reactQuery.useMutation)(['deleteSlo'], ({
    id
  }) => {
    try {
      return http.delete(`/api/observability/slos/${id}`);
    } catch (error) {
      return Promise.reject(`Something went wrong: ${String(error)}`);
    }
  }, {
    onMutate: async slo => {
      // Cancel any outgoing refetches (so they don't overwrite our optimistic update)
      await queryClient.cancelQueries(['fetchSloList'], {
        exact: false
      });
      const latestFetchSloListRequest = (queryClient.getQueriesData(['fetchSloList']) || []).at(0);
      const [queryKey, data] = latestFetchSloListRequest || [];
      const optimisticUpdate = {
        ...data,
        results: data === null || data === void 0 ? void 0 : data.results.filter(result => result.id !== slo.id),
        total: (data === null || data === void 0 ? void 0 : data.total) && data.total - 1
      };

      // Optimistically update to the new value
      if (queryKey) {
        queryClient.setQueryData(queryKey, optimisticUpdate);
      }
      toasts.addSuccess(_i18n.i18n.translate('xpack.observability.slo.slo.delete.successNotification', {
        defaultMessage: 'Deleted {name}',
        values: {
          name: slo.name
        }
      }));

      // Return a context object with the snapshotted value
      return {
        previousSloList: data
      };
    },
    // If the mutation fails, use the context returned from onMutate to roll back
    onError: (_err, slo, context) => {
      if (context !== null && context !== void 0 && context.previousSloList) {
        queryClient.setQueryData(['fetchSloList'], context.previousSloList);
      }
      toasts.addDanger(_i18n.i18n.translate('xpack.observability.slo.slo.delete.errorNotification', {
        defaultMessage: 'Failed to delete {name}',
        values: {
          name: slo.name
        }
      }));
    },
    onSuccess: () => {
      var _queryClient$getQuery;
      if (((_queryClient$getQuery = queryClient.getQueryCache().find(['fetchSloList'], {
        exact: false
      })) === null || _queryClient$getQuery === void 0 ? void 0 : _queryClient$getQuery.options // @ts-ignore
      .refetchInterval) === undefined) {
        queryClient.invalidateQueries(['fetchSloList'], {
          exact: false
        });
      }
    }
  });
  return deleteSlo;
}