"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useCreateSlo = useCreateSlo;
var _uuid = require("uuid");
var _reactQuery = require("@tanstack/react-query");
var _i18n = require("@kbn/i18n");
var _kibana_react = require("../../utils/kibana_react");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function useCreateSlo() {
  const {
    http,
    notifications: {
      toasts
    }
  } = (0, _kibana_react.useKibana)().services;
  const queryClient = (0, _reactQuery.useQueryClient)();
  return (0, _reactQuery.useMutation)(({
    slo
  }) => {
    const body = JSON.stringify(slo);
    return http.post(`/api/observability/slos`, {
      body
    });
  }, {
    mutationKey: ['createSlo'],
    onSuccess: (_data, {
      slo: {
        name
      }
    }) => {
      toasts.addSuccess(_i18n.i18n.translate('xpack.observability.slo.create.successNotification', {
        defaultMessage: 'Successfully created {name}',
        values: {
          name
        }
      }));
      queryClient.invalidateQueries(['fetchSloList']);
    },
    onError: (error, {
      slo: {
        name
      }
    }) => {
      toasts.addError(new Error(String(error)), {
        title: _i18n.i18n.translate('xpack.observability.slo.create.errorNotification', {
          defaultMessage: 'Something went wrong while creating {name}',
          values: {
            name
          }
        })
      });
    },
    onMutate: async ({
      slo
    }) => {
      // Cancel any outgoing refetches (so they don't overwrite our optimistic update)
      await queryClient.cancelQueries(['fetchSloList']);
      const latestFetchSloListRequest = (queryClient.getQueriesData(['fetchSloList']) || []).at(0);
      const [queryKey, data] = latestFetchSloListRequest || [];
      const newItem = {
        ...slo,
        id: (0, _uuid.v1)()
      };
      const optimisticUpdate = {
        ...data,
        results: [...((data === null || data === void 0 ? void 0 : data.results) || []), {
          ...newItem
        }],
        total: data !== null && data !== void 0 && data.total ? data.total + 1 : 1
      };

      // Optimistically update to the new value
      if (queryKey) {
        queryClient.setQueryData(queryKey, optimisticUpdate);
      }

      // Return a context object with the snapshotted value
      return {
        previousSloList: data
      };
    }
  });
}