"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useCloneSlo = useCloneSlo;
var _uuid = require("uuid");
var _reactQuery = require("@tanstack/react-query");
var _i18n = require("@kbn/i18n");
var _kibana_react = require("../../utils/kibana_react");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function useCloneSlo() {
  const {
    http,
    notifications: {
      toasts
    }
  } = (0, _kibana_react.useKibana)().services;
  const queryClient = (0, _reactQuery.useQueryClient)();
  return (0, _reactQuery.useMutation)(['cloneSlo'], ({
    slo
  }) => {
    const body = JSON.stringify(slo);
    return http.post(`/api/observability/slos`, {
      body
    });
  }, {
    onMutate: async ({
      slo,
      idToCopyFrom
    }) => {
      // Cancel any outgoing refetches (so they don't overwrite our optimistic update)
      await queryClient.cancelQueries(['fetchSloList'], {
        exact: false
      });
      const latestFetchSloListRequest = (queryClient.getQueriesData(['fetchSloList']) || []).at(0);
      const [queryKey, data] = latestFetchSloListRequest || [];
      const sloUsedToClone = data === null || data === void 0 ? void 0 : data.results.find(el => el.id === idToCopyFrom);
      const optimisticUpdate = {
        ...data,
        results: [...((data === null || data === void 0 ? void 0 : data.results) || []), {
          ...sloUsedToClone,
          name: slo.name,
          id: (0, _uuid.v1)(),
          summary: undefined
        }],
        total: (data === null || data === void 0 ? void 0 : data.total) && data.total + 1
      };

      // Optimistically update to the new value
      if (queryKey) {
        queryClient.setQueryData(queryKey, optimisticUpdate);
      }
      toasts.addSuccess(_i18n.i18n.translate('xpack.observability.slo.clone.successNotification', {
        defaultMessage: 'Successfully created {name}',
        values: {
          name: slo.name
        }
      }));

      // Return a context object with the snapshotted value
      return {
        previousSloList: data
      };
    },
    // If the mutation fails, use the context returned from onMutate to roll back
    onError: (_err, {
      slo
    }, context) => {
      if (context !== null && context !== void 0 && context.previousSloList) {
        queryClient.setQueryData(['fetchSloList'], context.previousSloList);
      }
      toasts.addDanger(_i18n.i18n.translate('xpack.observability.slo.clone.errorNotification', {
        defaultMessage: 'Failed to clone {name}',
        values: {
          name: slo.name
        }
      }));
    },
    onSuccess: () => {
      queryClient.invalidateQueries(['fetchSloList'], {
        exact: false
      });
    }
  });
}