"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getPipelineVertexStatsAggregation = getPipelineVertexStatsAggregation;
var _get_index_patterns = require("../cluster/get_index_patterns");
var _create_query = require("../create_query");
var _metrics = require("../metrics");
var _static_globals = require("../../static_globals");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function scalarCounterAggregation(field, fieldPath, ephemeralIdField, maxBucketSize) {
  const fullPath = `${fieldPath}.${field}`;
  const byEphemeralIdName = `${field}_temp_by_ephemeral_id`;
  const sumName = `${field}_total`;
  const aggs = {};
  aggs[byEphemeralIdName] = {
    terms: {
      field: ephemeralIdField,
      size: maxBucketSize
    },
    aggs: {
      stats: {
        stats: {
          field: fullPath
        }
      },
      difference: {
        bucket_script: {
          script: 'params.max - params.min',
          buckets_path: {
            min: 'stats.min',
            max: 'stats.max'
          }
        }
      }
    }
  };
  aggs[sumName] = {
    sum_bucket: {
      buckets_path: `${byEphemeralIdName}>difference`
    }
  };
  return aggs;
}
function createAggsObjectFromAggsList(aggsList) {
  return aggsList.reduce((aggsSoFar, agg) => ({
    ...aggsSoFar,
    ...agg
  }), {});
}
function createNestedVertexAgg(statsPath, vertexId, maxBucketSize) {
  const fieldPath = `${statsPath}.pipelines.vertices`;
  const ephemeralIdField = `${statsPath}.pipelines.vertices.pipeline_ephemeral_id`;
  return {
    vertices: {
      nested: {
        path: `${statsPath}.pipelines.vertices`
      },
      aggs: {
        vertex_id: {
          filter: {
            term: {
              [`${statsPath}.pipelines.vertices.id`]: vertexId
            }
          },
          aggs: {
            ...scalarCounterAggregation('events_in', fieldPath, ephemeralIdField, maxBucketSize),
            ...scalarCounterAggregation('events_out', fieldPath, ephemeralIdField, maxBucketSize),
            ...scalarCounterAggregation('duration_in_millis', fieldPath, ephemeralIdField, maxBucketSize),
            ...scalarCounterAggregation('queue_push_duration_in_millis', fieldPath, ephemeralIdField, maxBucketSize)
          }
        }
      }
    }
  };
}
function createTotalProcessorDurationStatsAgg(statsPath) {
  return {
    total_processor_duration_stats: {
      stats: {
        field: `${statsPath}.pipelines.events.duration_in_millis`
      }
    }
  };
}
function createScopedAgg(pipelineId, pipelineHash, vertexId, maxBucketSize) {
  return statsPath => {
    const aggs = {
      ...createNestedVertexAgg(statsPath, vertexId, maxBucketSize),
      ...createTotalProcessorDurationStatsAgg(statsPath)
    };
    return {
      nested: {
        path: `${statsPath}.pipelines`
      },
      aggs: {
        scoped: {
          filter: {
            bool: {
              filter: [{
                term: {
                  [`${statsPath}.pipelines.id`]: pipelineId
                }
              }, {
                term: {
                  [`${statsPath}.pipelines.hash`]: pipelineHash
                }
              }]
            }
          },
          aggs
        }
      }
    };
  };
}
function createTimeSeriesAgg(timeSeriesIntervalInSeconds, ...aggsList) {
  return {
    timeseries: {
      date_histogram: {
        field: 'timestamp',
        fixed_interval: timeSeriesIntervalInSeconds + 's'
      },
      aggs: createAggsObjectFromAggsList(aggsList)
    }
  };
}
function fetchPipelineVertexTimeSeriesStats({
  query,
  pipelineId,
  version,
  vertexId,
  timeSeriesIntervalInSeconds,
  maxBucketSize,
  callWithRequest,
  req
}) {
  const pipelineAggregation = createScopedAgg(pipelineId, version.hash, vertexId, maxBucketSize);
  const aggs = {
    ...createTimeSeriesAgg(timeSeriesIntervalInSeconds, {
      pipelines: pipelineAggregation('logstash_stats'),
      pipelines_mb: pipelineAggregation('logstash.node.stats')
    })
  };
  const indexPatterns = (0, _get_index_patterns.getIndexPatterns)({
    config: _static_globals.Globals.app.config,
    ccs: req.payload.ccs,
    moduleType: 'logstash'
  });
  const params = {
    index: indexPatterns,
    size: 0,
    ignore_unavailable: true,
    filter_path: ['aggregations.timeseries.buckets.key', 'aggregations.timeseries.buckets.pipelines.scoped.vertices.vertex_id.events_in_total', 'aggregations.timeseries.buckets.pipelines.scoped.vertices.vertex_id.events_out_total', 'aggregations.timeseries.buckets.pipelines.scoped.vertices.vertex_id.duration_in_millis_total', 'aggregations.timeseries.buckets.pipelines.scoped.vertices.vertex_id.queue_push_duration_in_millis_total', 'aggregations.timeseries.buckets.pipelines.scoped.total_processor_duration_stats', 'aggregations.timeseries.buckets.pipelines_mb.scoped.vertices.vertex_id.events_in_total', 'aggregations.timeseries.buckets.pipelines_mb.scoped.vertices.vertex_id.events_out_total', 'aggregations.timeseries.buckets.pipelines_mb.scoped.vertices.vertex_id.duration_in_millis_total', 'aggregations.timeseries.buckets.pipelines_mb.scoped.vertices.vertex_id.queue_push_duration_in_millis_total', 'aggregations.timeseries.buckets.pipelines_mb.scoped.total_processor_duration_stats'],
    body: {
      query,
      aggs
    }
  };
  return callWithRequest(req, 'search', params);
}
function getPipelineVertexStatsAggregation({
  req,
  timeSeriesIntervalInSeconds,
  clusterUuid,
  pipelineId,
  version,
  vertexId
}) {
  const {
    callWithRequest
  } = req.server.plugins.elasticsearch.getCluster('monitoring');
  const filters = [{
    bool: {
      should: [{
        nested: {
          path: 'logstash_stats.pipelines',
          ignore_unmapped: true,
          query: {
            bool: {
              filter: [{
                term: {
                  'logstash_stats.pipelines.id': pipelineId
                }
              }]
            }
          }
        }
      }, {
        nested: {
          path: 'logstash.node.stats.pipelines',
          ignore_unmapped: true,
          query: {
            bool: {
              filter: [{
                term: {
                  'logstash.node.stats.pipelines.id': pipelineId
                }
              }]
            }
          }
        }
      }]
    }
  }];
  const start = req.payload.timeRange.min;
  const end = req.payload.timeRange.max;
  const dataset = 'node_stats';
  const type = 'logstash_stats';
  const query = (0, _create_query.createQuery)({
    type,
    dsDataset: (0, _get_index_patterns.getLogstashDataset)(dataset),
    metricset: dataset,
    start,
    end,
    metric: _metrics.LogstashMetric.getMetricFields(),
    clusterUuid,
    filters
  });
  const config = req.server.config;
  return fetchPipelineVertexTimeSeriesStats({
    query,
    pipelineId,
    version,
    vertexId,
    timeSeriesIntervalInSeconds,
    maxBucketSize: config.ui.max_bucket_size,
    callWithRequest,
    req
  });
}