"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.JsonSchemaService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _fs = _interopRequireDefault(require("fs"));
var _path = _interopRequireDefault(require("path"));
var _schema_overrides = require("./schema_overrides");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const supportedEndpoints = [{
  path: '/_ml/anomaly_detectors/{job_id}',
  method: 'put'
}, {
  path: '/_ml/datafeeds/{datafeed_id}',
  method: 'put'
}];

/**
 *
 */
class JsonSchemaService {
  constructor() {
    /**
     * Dictionary with the schema components
     * @private
     */
    (0, _defineProperty2.default)(this, "_schemaComponents", {});
    (0, _defineProperty2.default)(this, "_requiredComponents", new Set());
  }
  /**
   * Extracts properties definition
   * @private
   */
  extractProperties(propertyDef) {
    for (const key in propertyDef) {
      if (propertyDef.hasOwnProperty(key)) {
        const propValue = propertyDef[key];
        if (key === '$ref') {
          const comp = propValue.split('/');
          const refKey = comp[comp.length - 1];
          delete propertyDef.$ref;

          // FIXME there is an issue with the maximum call stack size exceeded
          if (!refKey.startsWith('Ml_Types_')) return;
          const schemaComponent = this._schemaComponents[refKey];
          this._requiredComponents.add(refKey);
          Object.assign(propertyDef, schemaComponent);
          this.extractProperties(propertyDef);
        }
        if (Array.isArray(propValue)) {
          propValue.forEach(v => {
            if (typeof v === 'object') {
              this.extractProperties(v);
            }
          });
        } else if (typeof propValue === 'object') {
          this.extractProperties(propValue);
        }
      }
    }
  }
  applyOverrides(path, schema) {
    const overrides = _schema_overrides.jsonSchemaOverrides[path];
    return {
      ...schema,
      ...overrides,
      properties: {
        ...schema.properties,
        ...overrides.properties
      }
    };
  }

  /**
   * Extracts resolved schema definition for requested path and method
   * @param path
   * @param method
   */
  async extractSchema(path, method, schema) {
    const fileContent = schema !== null && schema !== void 0 ? schema : JSON.parse(_fs.default.readFileSync(_path.default.resolve(__dirname, 'openapi.json'), 'utf8'));
    const definition = fileContent.paths[path][method];
    if (!definition) {
      throw new Error('Schema definition is not defined');
    }
    const bodySchema = definition.requestBody.content['application/json'].schema;
    this._schemaComponents = fileContent.components.schemas;
    this.extractProperties(bodySchema);
    return this.applyOverrides(path, bodySchema);
  }

  /**
   * Generates openapi file, removing redundant content.
   * Only used internally via a node command to generate the file.
   */
  async generateSchemaFile() {
    const schema = JSON.parse(_fs.default.readFileSync(_path.default.resolve(__dirname, 'openapi_source.json'), 'utf8'));
    supportedEndpoints.forEach(e => {
      // need to extract schema in order to keep required components
      this.extractSchema(e.path, e.method, schema);
    });
    for (const pathName in schema.paths) {
      if (!schema.paths.hasOwnProperty(pathName)) continue;
      const supportedEndpoint = supportedEndpoints.find(v => v.path === pathName);
      if (supportedEndpoint) {
        for (const methodName in schema.paths[pathName]) {
          if (methodName !== supportedEndpoint.method) {
            delete schema.paths[pathName][methodName];
          }
        }
      } else {
        delete schema.paths[pathName];
      }
    }
    const components = schema.components.schemas;
    for (const componentName in components) {
      if (!this._requiredComponents.has(componentName)) {
        delete components[componentName];
      }
    }
    _fs.default.writeFileSync(_path.default.resolve(__dirname, 'openapi.json'), JSON.stringify(schema, null, 2));
  }
}
exports.JsonSchemaService = JsonSchemaService;