"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addFilterList = addFilterList;
exports.isValidFilterListId = isValidFilterListId;
exports.saveFilterList = saveFilterList;
exports.updateFilterList = updateFilterList;
var _i18n = require("@kbn/i18n");
var _dependency_cache = require("../../../util/dependency_cache");
var _job_utils = require("../../../../../common/util/job_utils");
var _ml_api_service = require("../../../services/ml_api_service");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function isValidFilterListId(id) {
  //  Filter List ID requires the same format as a Job ID, therefore isJobIdValid can be used
  return id !== undefined && id.length > 0 && (0, _job_utils.isJobIdValid)(id);
}

// Saves a filter list, running an update if the supplied loadedFilterList, holding the
// original filter list to which edits are being applied, is defined with a filter_id property.
function saveFilterList(filterId, description, items, loadedFilterList) {
  return new Promise((resolve, reject) => {
    if (loadedFilterList === undefined || loadedFilterList.filter_id === undefined) {
      // Create a new filter.
      addFilterList(filterId, description, items).then(newFilter => {
        resolve(newFilter);
      }).catch(error => {
        reject(error);
      });
    } else {
      // Edit to existing filter.
      updateFilterList(loadedFilterList, description, items).then(updatedFilter => {
        resolve(updatedFilter);
      }).catch(error => {
        reject(error);
      });
    }
  });
}
function addFilterList(filterId, description, items) {
  const filterWithIdExistsErrorMessage = _i18n.i18n.translate('xpack.ml.settings.filterLists.filterWithIdExistsErrorMessage', {
    defaultMessage: 'A filter with id {filterId} already exists',
    values: {
      filterId
    }
  });
  return new Promise((resolve, reject) => {
    // First check the filterId isn't already in use by loading the current list of filters.
    _ml_api_service.ml.filters.filtersStats().then(filterLists => {
      const savedFilterIds = filterLists.map(filterList => filterList.filter_id);
      if (savedFilterIds.indexOf(filterId) === -1) {
        // Save the new filter.
        _ml_api_service.ml.filters.addFilter(filterId, description, items).then(newFilter => {
          resolve(newFilter);
        }).catch(error => {
          reject(error);
        });
      } else {
        const toastNotifications = (0, _dependency_cache.getToastNotifications)();
        toastNotifications.addDanger(filterWithIdExistsErrorMessage);
        reject(new Error(filterWithIdExistsErrorMessage));
      }
    }).catch(error => {
      reject(error);
    });
  });
}
function updateFilterList(loadedFilterList, description, items) {
  return new Promise((resolve, reject) => {
    // Get items added and removed from loaded filter.
    const loadedItems = loadedFilterList.items;
    const addItems = items.filter(item => loadedItems.includes(item) === false);
    const removeItems = loadedItems.filter(item => items.includes(item) === false);
    _ml_api_service.ml.filters.updateFilter(loadedFilterList.filter_id, description, addItems, removeItems).then(updatedFilter => {
      resolve(updatedFilter);
    }).catch(error => {
      reject(error);
    });
  });
}