"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.trainedModelsApiProvider = trainedModelsApiProvider;
exports.useTrainedModelsApiService = useTrainedModelsApiService;
var _react = require("react");
var _ = require(".");
var _kibana = require("../../contexts/kibana");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Service with APIs calls to perform inference operations.
 * @param httpService
 */
function trainedModelsApiProvider(httpService) {
  const apiBasePath = (0, _.basePath)();
  return {
    /**
     * Fetches configuration information for a trained inference model.
     *
     * @param modelId - Model ID, collection of Model IDs or Model ID pattern.
     *                  Fetches all In case nothing is provided.
     * @param params - Optional query params
     */
    getTrainedModels(modelId, params) {
      const model = Array.isArray(modelId) ? modelId.join(',') : modelId;
      return httpService.http({
        path: `${apiBasePath}/trained_models${model ? `/${model}` : ''}`,
        method: 'GET',
        ...(params ? {
          query: params
        } : {})
      });
    },
    /**
     * Fetches usage information for trained inference models.
     *
     * @param modelId - Model ID, collection of Model IDs or Model ID pattern.
     *                  Fetches all In case nothing is provided.
     * @param params - Optional query params
     */
    getTrainedModelStats(modelId, params) {
      const model = Array.isArray(modelId) ? modelId.join(',') : modelId;
      return httpService.http({
        path: `${apiBasePath}/trained_models${model ? `/${model}` : ''}/_stats`,
        method: 'GET'
      });
    },
    /**
     * Fetches pipelines associated with provided models
     *
     * @param modelId - Model ID, collection of Model IDs.
     */
    getTrainedModelPipelines(modelId) {
      let model = modelId;
      if (Array.isArray(modelId)) {
        model = modelId.join(',');
      }
      return httpService.http({
        path: `${apiBasePath}/trained_models/${model}/pipelines`,
        method: 'GET'
      });
    },
    /**
     * Deletes an existing trained inference model.
     *
     * @param modelId - Model ID
     */
    deleteTrainedModel(modelId) {
      return httpService.http({
        path: `${apiBasePath}/trained_models/${modelId}`,
        method: 'DELETE'
      });
    },
    getTrainedModelsNodesOverview() {
      return httpService.http({
        path: `${apiBasePath}/model_management/nodes_overview`,
        method: 'GET'
      });
    },
    startModelAllocation(modelId, queryParams) {
      return httpService.http({
        path: `${apiBasePath}/trained_models/${modelId}/deployment/_start`,
        method: 'POST',
        query: queryParams
      });
    },
    stopModelAllocation(modelId, deploymentsIds, options = {
      force: false
    }) {
      const force = options === null || options === void 0 ? void 0 : options.force;
      return httpService.http({
        path: `${apiBasePath}/trained_models/${modelId}/${deploymentsIds.join(',')}/deployment/_stop`,
        method: 'POST',
        query: {
          force
        }
      });
    },
    updateModelDeployment(modelId, deploymentId, params) {
      return httpService.http({
        path: `${apiBasePath}/trained_models/${modelId}/${deploymentId}/deployment/_update`,
        method: 'POST',
        body: JSON.stringify(params)
      });
    },
    inferTrainedModel(modelId, deploymentsId, payload, timeout) {
      const body = JSON.stringify(payload);
      return httpService.http({
        path: `${apiBasePath}/trained_models/infer/${modelId}/${deploymentsId}`,
        method: 'POST',
        body,
        ...(timeout ? {
          query: {
            timeout
          }
        } : {})
      });
    },
    trainedModelPipelineSimulate(pipeline, docs) {
      const body = JSON.stringify({
        pipeline,
        docs
      });
      return httpService.http({
        path: `${apiBasePath}/trained_models/pipeline_simulate`,
        method: 'POST',
        body
      });
    },
    memoryUsage(type, node, showClosedJobs = false) {
      return httpService.http({
        path: `${apiBasePath}/model_management/memory_usage`,
        method: 'GET',
        query: {
          type,
          node,
          showClosedJobs
        }
      });
    },
    putTrainedModelConfig(modelId, config) {
      return httpService.http({
        path: `${apiBasePath}/trained_models/${modelId}`,
        method: 'PUT',
        body: JSON.stringify(config)
      });
    }
  };
}
/**
 * Hooks for accessing {@link TrainedModelsApiService} in React components.
 */
function useTrainedModelsApiService() {
  const {
    services: {
      mlServices: {
        httpService
      }
    }
  } = (0, _kibana.useMlKibana)();
  return (0, _react.useMemo)(() => trainedModelsApiProvider(httpService), [httpService]);
}