"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.basePath = basePath;
exports.ml = void 0;
exports.mlApiServicesProvider = mlApiServicesProvider;
var _json_schema = require("./json_schema");
var _http_service = require("../http_service");
var _annotations = require("./annotations");
var _data_frame_analytics = require("./data_frame_analytics");
var _filters = require("./filters");
var _results = require("./results");
var _jobs = require("./jobs");
var _saved_objects = require("./saved_objects");
var _trained_models = require("./trained_models");
var _dependency_cache = require("../../util/dependency_cache");
var _notifications = require("./notifications");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function basePath() {
  return '/api/ml';
}

/**
 * Temp solution to allow {@link ml} service to use http from
 * the dependency_cache.
 */
const proxyHttpStart = new Proxy({}, {
  get(obj, prop) {
    try {
      return (0, _dependency_cache.getHttp)()[prop];
    } catch (e) {
      if (prop === 'getLoadingCount$') {
        return () => {};
      }
      // eslint-disable-next-line no-console
      console.error(e);
    }
  }
});
const ml = mlApiServicesProvider(new _http_service.HttpService(proxyHttpStart));
exports.ml = ml;
function mlApiServicesProvider(httpService) {
  return {
    getJobs(obj) {
      const jobId = obj && obj.jobId ? `/${obj.jobId}` : '';
      return httpService.http({
        path: `${basePath()}/anomaly_detectors${jobId}`
      });
    },
    getJobStats(obj) {
      const jobId = obj && obj.jobId ? `/${obj.jobId}` : '';
      return httpService.http({
        path: `${basePath()}/anomaly_detectors${jobId}/_stats`
      });
    },
    addJob({
      jobId,
      job
    }) {
      const body = JSON.stringify(job);
      return httpService.http({
        path: `${basePath()}/anomaly_detectors/${jobId}`,
        method: 'PUT',
        body
      });
    },
    openJob({
      jobId
    }) {
      return httpService.http({
        path: `${basePath()}/anomaly_detectors/${jobId}/_open`,
        method: 'POST'
      });
    },
    closeJob({
      jobId
    }) {
      return httpService.http({
        path: `${basePath()}/anomaly_detectors/${jobId}/_close`,
        method: 'POST'
      });
    },
    forceCloseJob({
      jobId
    }) {
      return httpService.http({
        path: `${basePath()}/anomaly_detectors/${jobId}/_close?force=true`,
        method: 'POST'
      });
    },
    deleteJob({
      jobId
    }) {
      return httpService.http({
        path: `${basePath()}/anomaly_detectors/${jobId}`,
        method: 'DELETE'
      });
    },
    forceDeleteJob({
      jobId
    }) {
      return httpService.http({
        path: `${basePath()}/anomaly_detectors/${jobId}?force=true`,
        method: 'DELETE'
      });
    },
    updateJob({
      jobId,
      job
    }) {
      const body = JSON.stringify(job);
      return httpService.http({
        path: `${basePath()}/anomaly_detectors/${jobId}/_update`,
        method: 'POST',
        body
      });
    },
    resetJob({
      jobId
    }) {
      return httpService.http({
        path: `${basePath()}/anomaly_detectors/${jobId}/_reset`,
        method: 'POST'
      });
    },
    estimateBucketSpan(obj) {
      const body = JSON.stringify(obj);
      return httpService.http({
        path: `${basePath()}/validate/estimate_bucket_span`,
        method: 'POST',
        body
      });
    },
    validateJob(payload) {
      const body = JSON.stringify(payload);
      return httpService.http({
        path: `${basePath()}/validate/job`,
        method: 'POST',
        body
      });
    },
    validateDatafeedPreview(payload) {
      const body = JSON.stringify(payload);
      return httpService.http({
        path: `${basePath()}/validate/datafeed_preview`,
        method: 'POST',
        body
      });
    },
    validateCardinality$(job) {
      const body = JSON.stringify(job);
      return httpService.http$({
        path: `${basePath()}/validate/cardinality`,
        method: 'POST',
        body
      });
    },
    getDatafeeds(obj) {
      const datafeedId = obj && obj.datafeedId ? `/${obj.datafeedId}` : '';
      return httpService.http({
        path: `${basePath()}/datafeeds${datafeedId}`
      });
    },
    getDatafeedStats(obj) {
      const datafeedId = obj && obj.datafeedId ? `/${obj.datafeedId}` : '';
      return httpService.http({
        path: `${basePath()}/datafeeds${datafeedId}/_stats`
      });
    },
    addDatafeed({
      datafeedId,
      datafeedConfig
    }) {
      const body = JSON.stringify(datafeedConfig);
      return httpService.http({
        path: `${basePath()}/datafeeds/${datafeedId}`,
        method: 'PUT',
        body
      });
    },
    updateDatafeed({
      datafeedId,
      datafeedConfig
    }) {
      const body = JSON.stringify(datafeedConfig);
      return httpService.http({
        path: `${basePath()}/datafeeds/${datafeedId}/_update`,
        method: 'POST',
        body
      });
    },
    deleteDatafeed({
      datafeedId
    }) {
      return httpService.http({
        path: `${basePath()}/datafeeds/${datafeedId}`,
        method: 'DELETE'
      });
    },
    forceDeleteDatafeed({
      datafeedId
    }) {
      return httpService.http({
        path: `${basePath()}/datafeeds/${datafeedId}?force=true`,
        method: 'DELETE'
      });
    },
    startDatafeed({
      datafeedId,
      start,
      end
    }) {
      const body = JSON.stringify({
        ...(start !== undefined ? {
          start
        } : {}),
        ...(end !== undefined ? {
          end
        } : {})
      });
      return httpService.http({
        path: `${basePath()}/datafeeds/${datafeedId}/_start`,
        method: 'POST',
        body
      });
    },
    stopDatafeed({
      datafeedId
    }) {
      return httpService.http({
        path: `${basePath()}/datafeeds/${datafeedId}/_stop`,
        method: 'POST'
      });
    },
    forceStopDatafeed({
      datafeedId
    }) {
      return httpService.http({
        path: `${basePath()}/datafeeds/${datafeedId}/_stop?force=true`,
        method: 'POST'
      });
    },
    datafeedPreview({
      datafeedId
    }) {
      return httpService.http({
        path: `${basePath()}/datafeeds/${datafeedId}/_preview`,
        method: 'GET'
      });
    },
    validateDetector({
      detector
    }) {
      const body = JSON.stringify(detector);
      return httpService.http({
        path: `${basePath()}/anomaly_detectors/_validate/detector`,
        method: 'POST',
        body
      });
    },
    forecast({
      jobId,
      duration
    }) {
      const body = JSON.stringify({
        ...(duration !== undefined ? {
          duration
        } : {})
      });
      return httpService.http({
        path: `${basePath()}/anomaly_detectors/${jobId}/_forecast`,
        method: 'POST',
        body
      });
    },
    overallBuckets({
      jobId,
      topN,
      bucketSpan,
      start,
      end,
      overallScore
    }) {
      const body = JSON.stringify({
        topN,
        bucketSpan,
        start,
        end,
        ...(overallScore ? {
          overall_score: overallScore
        } : {})
      });
      return httpService.http({
        path: `${basePath()}/anomaly_detectors/${jobId}/results/overall_buckets`,
        method: 'POST',
        body
      });
    },
    hasPrivileges(obj) {
      const body = JSON.stringify(obj);
      return httpService.http({
        path: `${basePath()}/_has_privileges`,
        method: 'POST',
        body
      });
    },
    checkMlCapabilities() {
      return httpService.http({
        path: `${basePath()}/ml_capabilities`,
        method: 'GET'
      });
    },
    checkIndicesExists({
      indices
    }) {
      const body = JSON.stringify({
        indices
      });
      return httpService.http({
        path: `${basePath()}/index_exists`,
        method: 'POST',
        body
      });
    },
    getFieldCaps({
      index,
      fields
    }) {
      const body = JSON.stringify({
        ...(index !== undefined ? {
          index
        } : {}),
        ...(fields !== undefined ? {
          fields
        } : {})
      });
      return httpService.http({
        path: `${basePath()}/indices/field_caps`,
        method: 'POST',
        body
      });
    },
    recognizeIndex({
      indexPatternTitle
    }) {
      return httpService.http({
        path: `${basePath()}/modules/recognize/${indexPatternTitle}`,
        method: 'GET'
      });
    },
    listDataRecognizerModules() {
      return httpService.http({
        path: `${basePath()}/modules/get_module`,
        method: 'GET'
      });
    },
    getDataRecognizerModule({
      moduleId
    }) {
      return httpService.http({
        path: `${basePath()}/modules/get_module/${moduleId}`,
        method: 'GET'
      });
    },
    dataRecognizerModuleJobsExist({
      moduleId
    }) {
      return httpService.http({
        path: `${basePath()}/modules/jobs_exist/${moduleId}`,
        method: 'GET'
      });
    },
    setupDataRecognizerConfig({
      moduleId,
      prefix,
      groups,
      indexPatternName,
      query,
      useDedicatedIndex,
      startDatafeed,
      start,
      end,
      jobOverrides,
      estimateModelMemory
    }) {
      const body = JSON.stringify({
        prefix,
        groups,
        indexPatternName,
        query,
        useDedicatedIndex,
        startDatafeed,
        start,
        end,
        jobOverrides,
        estimateModelMemory
      });
      return httpService.http({
        path: `${basePath()}/modules/setup/${moduleId}`,
        method: 'POST',
        body
      });
    },
    getVisualizerFieldHistograms({
      indexPattern,
      query,
      fields,
      samplerShardSize,
      runtimeMappings
    }) {
      const body = JSON.stringify({
        query,
        fields,
        samplerShardSize,
        runtimeMappings
      });
      return httpService.http({
        path: `${basePath()}/data_visualizer/get_field_histograms/${indexPattern}`,
        method: 'POST',
        body
      });
    },
    /**
     * Gets a list of calendars
     * @param obj
     * @returns {Promise<Calendar[]>}
     */
    calendars(obj) {
      const {
        calendarId,
        calendarIds
      } = obj || {};
      let calendarIdsPathComponent = '';
      if (calendarId) {
        calendarIdsPathComponent = `/${calendarId}`;
      } else if (calendarIds) {
        calendarIdsPathComponent = `/${calendarIds.join(',')}`;
      }
      return httpService.http({
        path: `${basePath()}/calendars${calendarIdsPathComponent}`,
        method: 'GET'
      });
    },
    addCalendar(obj) {
      const body = JSON.stringify(obj);
      return httpService.http({
        path: `${basePath()}/calendars`,
        method: 'PUT',
        body
      });
    },
    updateCalendar(obj) {
      const calendarId = obj && obj.calendarId ? `/${obj.calendarId}` : '';
      const body = JSON.stringify(obj);
      return httpService.http({
        path: `${basePath()}/calendars${calendarId}`,
        method: 'PUT',
        body
      });
    },
    deleteCalendar({
      calendarId
    }) {
      return httpService.http({
        path: `${basePath()}/calendars/${calendarId}`,
        method: 'DELETE'
      });
    },
    mlNodeCount() {
      return httpService.http({
        path: `${basePath()}/ml_node_count`,
        method: 'GET'
      });
    },
    mlInfo() {
      return httpService.http({
        path: `${basePath()}/info`,
        method: 'GET'
      });
    },
    calculateModelMemoryLimit$({
      datafeedConfig,
      analysisConfig,
      indexPattern,
      query,
      timeFieldName,
      earliestMs,
      latestMs
    }) {
      const body = JSON.stringify({
        datafeedConfig,
        analysisConfig,
        indexPattern,
        query,
        timeFieldName,
        earliestMs,
        latestMs
      });
      return httpService.http$({
        path: `${basePath()}/validate/calculate_model_memory_limit`,
        method: 'POST',
        body
      });
    },
    getCardinalityOfFields({
      index,
      fieldNames,
      query,
      timeFieldName,
      earliestMs,
      latestMs
    }) {
      const body = JSON.stringify({
        index,
        fieldNames,
        query,
        timeFieldName,
        earliestMs,
        latestMs
      });
      return httpService.http({
        path: `${basePath()}/fields_service/field_cardinality`,
        method: 'POST',
        body
      });
    },
    getTimeFieldRange({
      index,
      timeFieldName,
      query,
      runtimeMappings,
      indicesOptions,
      allowFutureTime
    }) {
      const body = JSON.stringify({
        index,
        timeFieldName,
        query,
        runtimeMappings,
        indicesOptions,
        allowFutureTime
      });
      return httpService.http({
        path: `${basePath()}/fields_service/time_field_range`,
        method: 'POST',
        body
      });
    },
    esSearch(obj) {
      const body = JSON.stringify(obj);
      return httpService.http({
        path: `${basePath()}/es_search`,
        method: 'POST',
        body
      });
    },
    esSearch$(obj) {
      const body = JSON.stringify(obj);
      return httpService.http$({
        path: `${basePath()}/es_search`,
        method: 'POST',
        body
      });
    },
    getIndices() {
      const tempBasePath = '/api';
      return httpService.http({
        path: `${tempBasePath}/index_management/indices`,
        method: 'GET'
      });
    },
    getModelSnapshots(jobId, snapshotId) {
      return httpService.http({
        path: `${basePath()}/anomaly_detectors/${jobId}/model_snapshots${snapshotId !== undefined ? `/${snapshotId}` : ''}`
      });
    },
    updateModelSnapshot(jobId, snapshotId, body) {
      return httpService.http({
        path: `${basePath()}/anomaly_detectors/${jobId}/model_snapshots/${snapshotId}/_update`,
        method: 'POST',
        body: JSON.stringify(body)
      });
    },
    deleteModelSnapshot(jobId, snapshotId) {
      return httpService.http({
        path: `${basePath()}/anomaly_detectors/${jobId}/model_snapshots/${snapshotId}`,
        method: 'DELETE'
      });
    },
    annotations: (0, _annotations.annotationsApiProvider)(httpService),
    dataFrameAnalytics: (0, _data_frame_analytics.dataFrameAnalyticsApiProvider)(httpService),
    filters: (0, _filters.filtersApiProvider)(httpService),
    results: (0, _results.resultsApiProvider)(httpService),
    jobs: (0, _jobs.jobsApiProvider)(httpService),
    savedObjects: (0, _saved_objects.savedObjectsApiProvider)(httpService),
    trainedModels: (0, _trained_models.trainedModelsApiProvider)(httpService),
    notifications: (0, _notifications.notificationsProvider)(httpService),
    jsonSchema: (0, _json_schema.jsonSchemaProvider)(httpService)
  };
}