"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.dataFrameAnalyticsApiProvider = void 0;
exports.useDataFrameAnalyticsApiService = useDataFrameAnalyticsApiService;
var _react = require("react");
var _kibana = require("../../contexts/kibana");
var _ = require(".");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const dataFrameAnalyticsApiProvider = httpService => ({
  getDataFrameAnalytics(analyticsId, excludeGenerated, size) {
    const analyticsIdString = analyticsId !== undefined ? `/${analyticsId}` : '';
    return httpService.http({
      path: `${(0, _.basePath)()}/data_frame/analytics${analyticsIdString}`,
      method: 'GET',
      ...(excludeGenerated ? {
        query: {
          excludeGenerated,
          size
        }
      } : {})
    });
  },
  getDataFrameAnalyticsStats(analyticsId) {
    if (analyticsId !== undefined) {
      return httpService.http({
        path: `${(0, _.basePath)()}/data_frame/analytics/${analyticsId}/_stats`,
        method: 'GET'
      });
    }
    return httpService.http({
      path: `${(0, _.basePath)()}/data_frame/analytics/_stats`,
      method: 'GET'
    });
  },
  createDataFrameAnalytics(analyticsId, analyticsConfig) {
    const body = JSON.stringify(analyticsConfig);
    return httpService.http({
      path: `${(0, _.basePath)()}/data_frame/analytics/${analyticsId}`,
      method: 'PUT',
      body
    });
  },
  updateDataFrameAnalytics(analyticsId, updateConfig) {
    const body = JSON.stringify(updateConfig);
    return httpService.http({
      path: `${(0, _.basePath)()}/data_frame/analytics/${analyticsId}/_update`,
      method: 'POST',
      body
    });
  },
  getDataFrameAnalyticsMap(id, treatAsRoot, type) {
    const idString = id !== undefined ? `/${id}` : '';
    return httpService.http({
      path: `${(0, _.basePath)()}/data_frame/analytics/map${idString}`,
      method: 'GET',
      query: {
        treatAsRoot,
        type
      }
    });
  },
  jobsExist(analyticsIds, allSpaces = false) {
    const body = JSON.stringify({
      analyticsIds,
      allSpaces
    });
    return httpService.http({
      path: `${(0, _.basePath)()}/data_frame/analytics/jobs_exist`,
      method: 'POST',
      body
    });
  },
  evaluateDataFrameAnalytics(evaluateConfig) {
    const body = JSON.stringify(evaluateConfig);
    return httpService.http({
      path: `${(0, _.basePath)()}/data_frame/_evaluate`,
      method: 'POST',
      body
    });
  },
  explainDataFrameAnalytics(jobConfig) {
    const body = JSON.stringify(jobConfig);
    return httpService.http({
      path: `${(0, _.basePath)()}/data_frame/analytics/_explain`,
      method: 'POST',
      body
    });
  },
  deleteDataFrameAnalytics(analyticsId) {
    return httpService.http({
      path: `${(0, _.basePath)()}/data_frame/analytics/${analyticsId}`,
      method: 'DELETE'
    });
  },
  deleteDataFrameAnalyticsAndDestIndex(analyticsId, deleteDestIndex, deleteDestIndexPattern) {
    return httpService.http({
      path: `${(0, _.basePath)()}/data_frame/analytics/${analyticsId}`,
      query: {
        deleteDestIndex,
        deleteDestIndexPattern
      },
      method: 'DELETE'
    });
  },
  startDataFrameAnalytics(analyticsId) {
    return httpService.http({
      path: `${(0, _.basePath)()}/data_frame/analytics/${analyticsId}/_start`,
      method: 'POST'
    });
  },
  stopDataFrameAnalytics(analyticsId, force = false) {
    return httpService.http({
      path: `${(0, _.basePath)()}/data_frame/analytics/${analyticsId}/_stop`,
      method: 'POST',
      query: {
        force
      }
    });
  },
  getAnalyticsAuditMessages(analyticsId) {
    return httpService.http({
      path: `${(0, _.basePath)()}/data_frame/analytics/${analyticsId}/messages`,
      method: 'GET'
    });
  },
  validateDataFrameAnalytics(analyticsConfig) {
    const body = JSON.stringify(analyticsConfig);
    return httpService.http({
      path: `${(0, _.basePath)()}/data_frame/analytics/validate`,
      method: 'POST',
      body
    });
  },
  newJobCapsAnalytics(indexPatternTitle, isRollup = false) {
    const query = isRollup === true ? {
      rollup: true
    } : {};
    return httpService.http({
      path: `${(0, _.basePath)()}/data_frame/analytics/new_job_caps/${indexPatternTitle}`,
      method: 'GET',
      query
    });
  }
});
exports.dataFrameAnalyticsApiProvider = dataFrameAnalyticsApiProvider;
/**
 * Hooks for accessing {@link DataFrameAnalyticsApiService} in React components.
 */
function useDataFrameAnalyticsApiService() {
  const {
    services: {
      mlServices: {
        httpService
      }
    }
  } = (0, _kibana.useMlKibana)();
  return (0, _react.useMemo)(() => dataFrameAnalyticsApiProvider(httpService), [httpService]);
}