"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useResolver = void 0;
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _public = require("@kbn/saved-search-plugin/public");
var _index_utils = require("../util/index_utils");
var _new_job_utils = require("../jobs/new_job/utils/new_job_utils");
var _kibana = require("../contexts/kibana");
var _use_create_url = require("../contexts/kibana/use_create_url");
var _locator = require("../../../common/constants/locator");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Hook to resolve route specific requirements
 * @param dataViewId optional Kibana data view id, used for wizards
 * @param savedSearchId optional Kibana saved search id, used for wizards
 * @param config Kibana UI Settings
 * @param resolvers an array of resolvers to be executed for the route
 * @return { context, results } returns the ML context and resolver results
 */
const useResolver = (dataViewId, savedSearchId, config, dataViewsContract, getSavedSearchDeps, resolvers) => {
  const notifications = (0, _kibana.useNotifications)();
  const funcNames = Object.keys(resolvers); // Object.entries gets this wrong?!
  const funcs = Object.values(resolvers); // Object.entries gets this wrong?!
  const tempResults = funcNames.reduce((p, c) => {
    p[c] = {};
    return p;
  }, {});
  const [context, setContext] = (0, _react.useState)(null);
  const [results, setResults] = (0, _react.useState)(tempResults);
  const redirectToJobsManagementPage = (0, _use_create_url.useCreateAndNavigateToMlLink)(_locator.ML_PAGES.ANOMALY_DETECTION_JOBS_MANAGE);
  (0, _react.useEffect)(() => {
    (async () => {
      try {
        const res = await Promise.all(funcs.map(r => r()));
        res.forEach((r, i) => tempResults[funcNames[i]] = r);
        setResults(tempResults);
      } catch (error) {
        // quietly fail. Errors being thrown here are expected as a way to handle privilege or license check failures.
        // The user will be redirected by the failed resolver.
        return;
      }
      try {
        if (dataViewId === '') {
          throw new Error(_i18n.i18n.translate('xpack.ml.useResolver.errorIndexPatternIdEmptyString', {
            defaultMessage: 'dataViewId must not be empty string.'
          }));
        }
        let dataViewAndSavedSearch = {
          savedSearch: null,
          dataView: null
        };
        let savedSearch = null;
        if (savedSearchId !== undefined) {
          savedSearch = await (0, _public.getSavedSearch)(savedSearchId, getSavedSearchDeps);
          dataViewAndSavedSearch = await (0, _index_utils.getDataViewAndSavedSearch)(savedSearchId);
        } else if (dataViewId !== undefined) {
          dataViewAndSavedSearch.dataView = await (0, _index_utils.getDataViewById)(dataViewId);
        }
        const {
          savedSearch: deprecatedSavedSearchObj,
          dataView
        } = dataViewAndSavedSearch;
        const {
          combinedQuery
        } = (0, _new_job_utils.createSearchItems)(config, dataView !== null ? dataView : undefined, deprecatedSavedSearchObj);
        setContext({
          combinedQuery,
          currentDataView: dataView,
          deprecatedSavedSearchObj,
          selectedSavedSearch: savedSearch,
          dataViewsContract,
          kibanaConfig: config
        });
      } catch (error) {
        // an unexpected error has occurred. This could be caused by an incorrect index pattern or saved search ID
        notifications.toasts.addError(new Error(error), {
          title: _i18n.i18n.translate('xpack.ml.useResolver.errorTitle', {
            defaultMessage: 'An error has occurred'
          })
        });
        await redirectToJobsManagementPage();
      }
    })();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);
  return {
    context,
    results
  };
};
exports.useResolver = useResolver;