"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.checkForSavedObjects = void 0;
exports.checkViewOrCreateJobs = checkViewOrCreateJobs;
var _i18n = require("@kbn/i18n");
var _dependency_cache = require("../../../util/dependency_cache");
var _ml_api_service = require("../../../services/ml_api_service");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Checks whether the jobs in a data recognizer module have been created.
 * Redirects to the Anomaly Explorer to view the jobs if they have been created,
 * or the recognizer job wizard for the module if not.
 */
function checkViewOrCreateJobs(moduleId, dataViewId, createLinkWithUserDefaults, navigateToPath) {
  return new Promise((resolve, reject) => {
    // Load the module, and check if the job(s) in the module have been created.
    // If so, load the jobs in the Anomaly Explorer.
    // Otherwise open the data recognizer wizard for the module.
    // Always want to call reject() so as not to load original page.
    _ml_api_service.ml.dataRecognizerModuleJobsExist({
      moduleId
    }).then(async resp => {
      if (resp.jobsExist === true) {
        // also honor user's time filter setting in Advanced Settings
        const url = createLinkWithUserDefaults('explorer', resp.jobs);
        await navigateToPath(url);
        reject();
      } else {
        await navigateToPath(`/jobs/new_job/recognize?id=${moduleId}&index=${dataViewId}`);
        reject();
      }
    }).catch(async err => {
      // eslint-disable-next-line no-console
      console.error(`Error checking whether jobs in module ${moduleId} exists`, err);
      const toastNotifications = (0, _dependency_cache.getToastNotifications)();
      toastNotifications.addWarning({
        title: _i18n.i18n.translate('xpack.ml.newJob.recognize.moduleCheckJobsExistWarningTitle', {
          defaultMessage: 'Error checking module {moduleId}',
          values: {
            moduleId
          }
        }),
        text: _i18n.i18n.translate('xpack.ml.newJob.recognize.moduleCheckJobsExistWarningDescription', {
          defaultMessage: 'An error occurred trying to check whether the jobs in the module have been created.'
        })
      });
      await navigateToPath(`/jobs`);
      reject();
    });
  });
}

/**
 * Gets kibana objects with an existence check.
 */
const checkForSavedObjects = async objects => {
  const savedObjectsClient = (0, _dependency_cache.getSavedObjectsClient)();
  try {
    return await Object.keys(objects).reduce(async (prevPromise, type) => {
      const acc = await prevPromise;
      const {
        savedObjects
      } = await savedObjectsClient.find({
        type,
        perPage: 1000
      });
      acc[type] = objects[type].map(obj => {
        const find = savedObjects.find(savedObject => savedObject.attributes.title === obj.title);
        return {
          ...obj,
          exists: !!find,
          id: !!find && find.id || obj.id
        };
      });
      return Promise.resolve(acc);
    }, Promise.resolve({}));
  } catch (e) {
    // eslint-disable-next-line no-console
    console.error('Could not load saved objects', e);
  }
  return Promise.resolve(objects);
};
exports.checkForSavedObjects = checkForSavedObjects;