"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.VisualizationExtractor = void 0;
var _public = require("@kbn/lens-plugin/public");
var _i18n = require("@kbn/i18n");
var _new_job = require("../../../../../common/constants/new_job");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class VisualizationExtractor {
  constructor() {}
  async getResultLayersFromEmbeddable(embeddable, lens) {
    const {
      chartInfo
    } = await (0, _utils.getJobsItemsFromEmbeddable)(embeddable, lens);
    return this.getLayers(chartInfo, lens);
  }
  async extractFields(layer) {
    var _timeField$operation$;
    if (!(0, _utils.isCompatibleLayer)(layer)) {
      throw Error(_i18n.i18n.translate('xpack.ml.newJob.fromLens.createJob.error.incompatibleLayerType', {
        defaultMessage: 'Layer is incompatible. Only chart layers can be used.'
      }));
    }
    const timeField = layer.dimensions.find(dimension => {
      var _dimension$operation;
      return ((_dimension$operation = dimension.operation) === null || _dimension$operation === void 0 ? void 0 : _dimension$operation.dataType) === 'date';
    });
    if (timeField === undefined || !((_timeField$operation$ = timeField.operation.fields) !== null && _timeField$operation$ !== void 0 && _timeField$operation$.length)) {
      throw Error(_i18n.i18n.translate('xpack.ml.newJob.fromLens.createJob.error.noDateField', {
        defaultMessage: 'Cannot find a date field.'
      }));
    }
    const metricFields = layer.dimensions.filter(dimension => dimension.role === 'metric');
    validateDimensions(metricFields);
    const splitField = layer.dimensions.find(dimension => dimension.role === 'split' && dimension.dimensionType === 'breakdown');
    if (splitField && (0, _utils.isTermsField)(splitField) && splitField.operation.fields && splitField.operation.fields.length > 1) {
      throw Error(_i18n.i18n.translate('xpack.ml.newJob.fromLens.createJob.error.splitFieldHasMultipleFields', {
        defaultMessage: 'Selected split field contains more than one field.'
      }));
    }
    if (splitField && !(0, _utils.isCompatibleSplitFieldType)(splitField)) {
      throw Error(_i18n.i18n.translate('xpack.ml.newJob.fromLens.createJob.error.splitFieldMustBeString', {
        defaultMessage: 'Selected split field type must be string.'
      }));
    }
    if (!layer.dataView) {
      throw Error(_i18n.i18n.translate('xpack.ml.newJob.fromLens.createJob.error.noDataViews', {
        defaultMessage: 'No data views can be found in the visualization.'
      }));
    }
    if (timeField.operation.fields[0] !== layer.dataView.timeFieldName) {
      throw Error(_i18n.i18n.translate('xpack.ml.newJob.fromLens.createJob.error.timeFieldNotInDataView', {
        defaultMessage: 'Selected time field must be the default time field configured for data view.'
      }));
    }
    return {
      fields: metricFields,
      timeField,
      splitField,
      dataView: layer.dataView
    };
  }
  async getLayers(chartInfo, lens) {
    const getVisType = await (0, _utils.getVisTypeFactory)(lens);
    const layers = await Promise.all(chartInfo.layers.filter(({
      layerType
    }) => layerType === _public.layerTypes.DATA) // remove non chart layers
    .map(async layer => {
      const {
        icon,
        label
      } = getVisType(layer);
      try {
        const {
          fields,
          splitField
        } = await this.extractFields(layer);
        const detectors = (0, _utils.createDetectors)(fields, splitField);
        const jobType = splitField || detectors.length > 1 ? _new_job.JOB_TYPE.MULTI_METRIC : _new_job.JOB_TYPE.SINGLE_METRIC;
        return {
          id: layer.layerId,
          layerType: layer.layerType,
          label,
          icon,
          jobType,
          isCompatible: true
        };
      } catch (error) {
        return {
          id: layer.layerId,
          layerType: layer.layerType,
          label,
          icon,
          jobType: null,
          isCompatible: false,
          error
        };
      }
    }));
    return layers;
  }
}
exports.VisualizationExtractor = VisualizationExtractor;
function validateDimensions(dimensions) {
  dimensions.forEach(dimension => {
    // fail early if any of the cols being used as accessors
    // contain functions we don't support
    return dimension.operation.dataType !== 'date' && (0, _utils.getMlFunction)(dimension.operation.type);
  });
  if (dimensions.some(dimension => {
    var _dimension$operation$;
    return !((_dimension$operation$ = dimension.operation.fields) !== null && _dimension$operation$ !== void 0 && _dimension$operation$.length);
  })) {
    throw Error(_i18n.i18n.translate('xpack.ml.newJob.fromLens.createJob.error.colsNoSourceField', {
      defaultMessage: 'Some columns do not contain a source field.'
    }));
  }
  if (dimensions.some(dimension => (0, _utils.hasIncompatibleProperties)(dimension))) {
    throw Error(_i18n.i18n.translate('xpack.ml.newJob.fromLens.createJob.error.colsUsingFilterTimeSift', {
      defaultMessage: 'Columns contain settings which are incompatible with ML detectors, time shift and filter by are not supported.'
    }));
  }
}