"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getScoresByRecord = getScoresByRecord;
var _lodash = require("lodash");
var _string_utils = require("../../../../util/string_utils");
var _ml_api_service = require("../../../../services/ml_api_service");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// detector swimlane search
function getScoresByRecord(jobId, earliestMs, latestMs, intervalMs, firstSplitField) {
  return new Promise((resolve, reject) => {
    const obj = {
      success: true,
      results: {},
      totalResults: 0
    };
    let jobIdFilterStr = 'job_id: ' + jobId;
    if (firstSplitField && firstSplitField.value !== undefined) {
      // Escape any reserved characters for the query_string query,
      // wrapping the value in quotes to do a phrase match.
      // Backslash is a special character in JSON strings, so doubly escape
      // any backslash characters which exist in the field value.
      jobIdFilterStr += ` AND ${(0, _string_utils.escapeForElasticsearchQuery)(firstSplitField.name)}:`;
      jobIdFilterStr += `"${String(firstSplitField.value).replace(/\\/g, '\\\\')}"`;
    }
    _ml_api_service.ml.results.anomalySearch({
      body: {
        size: 0,
        query: {
          bool: {
            filter: [{
              query_string: {
                query: 'result_type:record'
              }
            }, {
              bool: {
                must: [{
                  range: {
                    timestamp: {
                      gte: earliestMs,
                      lte: latestMs,
                      // @ts-ignore
                      format: 'epoch_millis'
                    }
                  }
                }, {
                  query_string: {
                    query: jobIdFilterStr
                  }
                }]
              }
            }]
          }
        },
        aggs: {
          detector_index: {
            terms: {
              field: 'detector_index',
              order: {
                recordScore: 'desc'
              }
            },
            aggs: {
              recordScore: {
                max: {
                  field: 'record_score'
                }
              },
              byTime: {
                date_histogram: {
                  field: 'timestamp',
                  fixed_interval: `${intervalMs}ms`,
                  min_doc_count: 1,
                  extended_bounds: {
                    min: earliestMs,
                    max: latestMs
                  }
                },
                aggs: {
                  recordScore: {
                    max: {
                      field: 'record_score'
                    }
                  }
                }
              }
            }
          }
        }
      }
    }, [jobId]).then(resp => {
      const detectorsByIndex = (0, _lodash.get)(resp, ['aggregations', 'detector_index', 'buckets'], []);
      detectorsByIndex.forEach(dtr => {
        const dtrResults = [];
        const dtrIndex = +dtr.key;
        const buckets = (0, _lodash.get)(dtr, ['byTime', 'buckets'], []);
        for (let j = 0; j < buckets.length; j++) {
          const bkt = buckets[j];
          const time = bkt.key;
          dtrResults.push({
            time,
            value: (0, _lodash.get)(bkt, ['recordScore', 'value'])
          });
        }
        obj.results[dtrIndex] = dtrResults;
      });
      resolve(obj);
    }).catch(resp => {
      reject(resp);
    });
  });
}