"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ResultsLoader = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _job_creator = require("../job_creator");
var _results_service = require("../../../../services/results_service");
var _anomaly_utils = require("../../../../../../common/util/anomaly_utils");
var _parse_interval = require("../../../../../../common/util/parse_interval");
var _searches = require("./searches");
var _new_job = require("../../../../../../common/constants/new_job");
var _aggregation_types = require("../../../../../../common/constants/aggregation_types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const emptyModelItem = {
  time: 0,
  actual: 0,
  modelUpper: 0,
  modelLower: 0
};
const LAST_UPDATE_DELAY_MS = 500;
class ResultsLoader {
  constructor(jobCreator, chartInterval, chartLoader) {
    (0, _defineProperty2.default)(this, "_results$", void 0);
    (0, _defineProperty2.default)(this, "_resultsSearchRunning", false);
    (0, _defineProperty2.default)(this, "_jobCreator", void 0);
    (0, _defineProperty2.default)(this, "_chartInterval", void 0);
    (0, _defineProperty2.default)(this, "_lastModelTimeStamp", 0);
    (0, _defineProperty2.default)(this, "_lastResultsTimeout", null);
    (0, _defineProperty2.default)(this, "_chartLoader", void 0);
    (0, _defineProperty2.default)(this, "_results", {
      progress: 0,
      model: [],
      anomalies: []
    });
    (0, _defineProperty2.default)(this, "_detectorSplitFieldFilters", null);
    (0, _defineProperty2.default)(this, "_splitFieldFiltersLoaded", false);
    (0, _defineProperty2.default)(this, "progressSubscriber", async progress => {
      if (this._resultsSearchRunning === false && (progress - this._results.progress > 5 || progress === 100)) {
        if (this._splitFieldFiltersLoaded === false) {
          this._splitFieldFiltersLoaded = true;
          // load detector field filters if this is the first run.
          await this._populateDetectorSplitFieldFilters();
        }
        this._updateData(progress, false);
        if (progress === 100) {
          // after the job has finished, do one final update
          // a while after the last 100% has been received.
          // note, there may be multiple 100% progresses sent as they will only stop once the
          // datafeed has stopped.
          clearTimeout(this._lastResultsTimeout);
          this._lastResultsTimeout = setTimeout(() => {
            this._updateData(progress, true);
          }, LAST_UPDATE_DELAY_MS);
        }
      }
    });
    this._jobCreator = jobCreator;
    this._chartInterval = chartInterval;
    this._results$ = new _rxjs.BehaviorSubject(this._results);
    this._chartLoader = chartLoader;
    jobCreator.subscribeToProgress(this.progressSubscriber);
  }
  async _updateData(progress, fullRefresh) {
    this._resultsSearchRunning = true;
    if (fullRefresh === true) {
      this._clearResults();
    }
    this._results.progress = progress;
    const getAnomalyData = this._jobCreator.type === _new_job.JOB_TYPE.SINGLE_METRIC ? () => this._loadJobAnomalyData(0) : () => this._loadDetectorsAnomalyData();

    // TODO - load more that one model
    const [model, anomalies] = await Promise.all([this._loadModelData(0), getAnomalyData()]);
    this._results.model = model;
    this._results.anomalies = anomalies;
    this._resultsSearchRunning = false;
    this._results$.next(this._results);
  }
  get results$() {
    return this._results$;
  }
  subscribeToResults(func) {
    return this._results$.subscribe(func);
  }
  get progress() {
    return this._results.progress;
  }
  _clearResults() {
    this._results.model = {};
    this._results.anomalies = {};
    this._results.progress = 0;
    this._lastModelTimeStamp = 0;
  }
  async _loadModelData(dtrIndex) {
    if (this._jobCreator.modelPlot === false) {
      return [];
    }
    const agg = this._jobCreator.getAggregation(dtrIndex);
    if (agg === null) {
      return {
        [dtrIndex]: [emptyModelItem]
      };
    }
    const resp = await (0, _rxjs.lastValueFrom)(_results_service.mlResultsService.getModelPlotOutput(this._jobCreator.jobId, dtrIndex, [], this._lastModelTimeStamp, this._jobCreator.end, this._chartInterval.getInterval().asMilliseconds(), agg.mlModelPlotAgg));
    return this._createModel(resp, dtrIndex);
  }
  _createModel({
    results
  }, dtrIndex) {
    if (this._results.model[dtrIndex] === undefined) {
      this._results.model[dtrIndex] = [];
    }
    const dslName = this._jobCreator.aggregations[dtrIndex].dslName;
    const isCountAgg = dslName === _aggregation_types.ES_AGGREGATION.COUNT || dslName === _aggregation_types.ES_AGGREGATION.CARDINALITY;
    // if aggregation is count or distinct count, scale the model plot to match the real data.
    const modelScale = isCountAgg ? this._calculateModelScale() : 1;

    // create ModelItem list from search results
    const model = Object.entries(results).map(([time, {
      actual,
      modelLower,
      modelUpper
    }]) => ({
      time: +time,
      actual,
      modelUpper: (modelUpper === null ? 0 : modelUpper) * modelScale,
      modelLower: (modelLower === null ? 0 : modelLower) * modelScale
    }));
    if (model.length > 10) {
      // discard the last 5 buckets in the previously loaded model to avoid partial results
      // set the _lastModelTimeStamp to be 5 buckets behind so we load the correct
      // section of results next time.
      this._lastModelTimeStamp = model[model.length - 5].time;
      for (let i = 0; i < 5; i++) {
        this._results.model[dtrIndex].pop();
      }
    }

    // return a new array from the old and new model
    return {
      [dtrIndex]: this._results.model[dtrIndex].concat(model)
    };
  }
  async _loadJobAnomalyData(dtrIndex) {
    const resp = await _results_service.mlResultsService.getScoresByBucket([this._jobCreator.jobId], this._jobCreator.start, this._jobCreator.end, this._chartInterval.getInterval().asMilliseconds(), 1);
    const results = resp.results[this._jobCreator.jobId];
    if (results === undefined) {
      return [];
    }
    const anomalies = {};
    anomalies[0] = Object.entries(results).map(([time, value]) => ({
      time: +time,
      value,
      severity: (0, _anomaly_utils.getSeverityType)(value)
    }));
    return anomalies;
  }
  async _loadDetectorsAnomalyData() {
    const resp = await (0, _searches.getScoresByRecord)(this._jobCreator.jobId, this._jobCreator.start, this._jobCreator.end, this._chartInterval.getInterval().asMilliseconds(), this._detectorSplitFieldFilters);
    const anomalies = {};
    Object.entries(resp.results).forEach(([dtrIdx, results]) => {
      anomalies[+dtrIdx] = results.map(r => ({
        ...r,
        severity: (0, _anomaly_utils.getSeverityType)(r.value)
      }));
    });
    return anomalies;
  }
  async _populateDetectorSplitFieldFilters() {
    // only apply a split filter for multi-metric jobs
    // for population jobs, the swimlane will not be 100% accurate as it's not
    // filtering for the front card. this is a trade off to save loading
    // lots of data.
    if ((0, _job_creator.isMultiMetricJobCreator)(this._jobCreator)) {
      if (this._jobCreator.splitField !== null) {
        const fieldValues = await this._chartLoader.loadFieldExampleValues(this._jobCreator.splitField, this._jobCreator.runtimeMappings, this._jobCreator.datafeedConfig.indices_options);
        if (fieldValues.length > 0) {
          this._detectorSplitFieldFilters = {
            name: this._jobCreator.splitField.name,
            value: fieldValues[0]
          };
        }
        return;
      }
    }
    this._detectorSplitFieldFilters = null;
  }

  // calculate a scale for the model upper and lower by the ratio of chart interval to bucketspan.
  // this will force the model bounds to be drawn in the correct location
  _calculateModelScale() {
    const duration = (0, _parse_interval.parseInterval)(this._jobCreator.bucketSpan);
    const bucketSpanMs = duration !== null ? duration.asMilliseconds() : 0;
    const chartIntervalMs = this._chartInterval.getInterval().asMilliseconds();
    return chartIntervalMs / bucketSpanMs;
  }
}
exports.ResultsLoader = ResultsLoader;