"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.VALIDATION_DELAY_MS = exports.JobValidator = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _operators = require("rxjs/operators");
var _job_utils = require("../../../../../../common/util/job_utils");
var _ml_server_info = require("../../../../services/ml_server_info");
var _job_creator = require("../job_creator");
var _util = require("./util");
var _validators = require("./validators");
var _categorization_job = require("../../../../../../common/constants/categorization_job");
var _new_job = require("../../../../../../common/constants/new_job");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// delay start of validation to allow the user to make changes
// e.g. if they are typing in a new value, try not to validate
// after every keystroke
const VALIDATION_DELAY_MS = 500;
exports.VALIDATION_DELAY_MS = VALIDATION_DELAY_MS;
class JobValidator {
  /**
   * Observable that combines basic and async validation results.
   */

  constructor(jobCreator) {
    (0, _defineProperty2.default)(this, "_jobCreator", void 0);
    (0, _defineProperty2.default)(this, "_validationSummary", void 0);
    (0, _defineProperty2.default)(this, "_lastJobConfig", void 0);
    (0, _defineProperty2.default)(this, "_lastDatafeedConfig", void 0);
    (0, _defineProperty2.default)(this, "_validateTimeout", null);
    (0, _defineProperty2.default)(this, "_asyncValidators$", []);
    (0, _defineProperty2.default)(this, "_asyncValidatorsResult$", void 0);
    (0, _defineProperty2.default)(this, "_basicValidations", {
      jobId: {
        valid: true
      },
      groupIds: {
        valid: true
      },
      modelMemoryLimit: {
        valid: true
      },
      bucketSpan: {
        valid: true
      },
      duplicateDetectors: {
        valid: true
      },
      query: {
        valid: true
      },
      queryDelay: {
        valid: true
      },
      frequency: {
        valid: true
      },
      scrollSize: {
        valid: true
      },
      categorizerMissingPerPartition: {
        valid: true
      },
      categorizerVaryingPerPartitionField: {
        valid: true
      },
      summaryCountField: {
        valid: true
      }
    });
    (0, _defineProperty2.default)(this, "_advancedValidations", {
      categorizationFieldValid: {
        valid: true
      }
    });
    (0, _defineProperty2.default)(this, "_validating", false);
    (0, _defineProperty2.default)(this, "_basicValidationResult$", new _rxjs.ReplaySubject(2));
    (0, _defineProperty2.default)(this, "_jobCreatorSubject$", new _rxjs.Subject());
    (0, _defineProperty2.default)(this, "validationResult$", void 0);
    (0, _defineProperty2.default)(this, "latestValidationResult", this._basicValidations);
    this._jobCreator = jobCreator;
    this._lastJobConfig = this._jobCreator.formattedJobJson;
    this._lastDatafeedConfig = this._jobCreator.formattedDatafeedJson;
    this._validationSummary = {
      basic: false,
      advanced: false
    };
    this._asyncValidators$ = [(0, _validators.cardinalityValidator)(this._jobCreatorSubject$), (0, _validators.jobIdValidator)(this._jobCreatorSubject$), (0, _validators.groupIdsValidator)(this._jobCreatorSubject$)];
    this._asyncValidatorsResult$ = (0, _rxjs.combineLatest)(this._asyncValidators$).pipe((0, _operators.map)(res => {
      return res.reduce((acc, curr) => {
        return {
          ...acc,
          ...(curr ? curr : {})
        };
      }, {});
    }), (0, _operators.startWith)({}));
    this.validationResult$ = (0, _rxjs.combineLatest)([this._basicValidationResult$, this._asyncValidatorsResult$]).pipe((0, _operators.map)(([basicValidationResult, asyncValidatorsResult]) => {
      return {
        ...basicValidationResult,
        ...asyncValidatorsResult
      };
    }), (0, _operators.tap)(latestValidationResult => {
      this.latestValidationResult = latestValidationResult;
    }));
  }
  validate(callback, forceValidate = false) {
    this._validating = true;
    const formattedJobConfig = this._jobCreator.formattedJobJson;
    const formattedDatafeedConfig = this._jobCreator.formattedDatafeedJson;
    this._runAdvancedValidation();
    // only validate if the config has changed
    if (forceValidate || formattedJobConfig !== this._lastJobConfig || formattedDatafeedConfig !== this._lastDatafeedConfig) {
      if (this._validateTimeout !== null) {
        // clear any previous on going validation timeouts
        clearTimeout(this._validateTimeout);
      }
      this._lastJobConfig = formattedJobConfig;
      this._lastDatafeedConfig = formattedDatafeedConfig;
      this._validateTimeout = setTimeout(() => {
        this._runBasicValidation();
        this._jobCreatorSubject$.next(this._jobCreator);
        this._validating = false;
        this._validateTimeout = null;
        callback();
      }, VALIDATION_DELAY_MS);
    } else {
      // _validating is still true if there is a previous validation timeout on going.
      this._validating = this._validateTimeout !== null;
    }
    callback();
  }
  _resetBasicValidations() {
    this._validationSummary.basic = true;
    Object.values(this._basicValidations).forEach(v => {
      v.valid = true;
      delete v.message;
    });
  }
  _runBasicValidation() {
    this._resetBasicValidations();
    const jobConfig = this._jobCreator.jobConfig;
    const datafeedConfig = this._jobCreator.datafeedConfig;
    const limits = (0, _ml_server_info.getNewJobLimits)();

    // run standard basic validation
    const basicJobResults = (0, _job_utils.basicJobValidation)(jobConfig, undefined, limits);
    (0, _util.populateValidationMessages)(basicJobResults, this._basicValidations, jobConfig, datafeedConfig);
    const basicDatafeedResults = (0, _job_utils.basicDatafeedValidation)(datafeedConfig);
    (0, _util.populateValidationMessages)(basicDatafeedResults, this._basicValidations, jobConfig, datafeedConfig);
    const basicJobAndDatafeedResults = (0, _job_utils.basicJobAndDatafeedValidation)(jobConfig, datafeedConfig);
    (0, _util.populateValidationMessages)(basicJobAndDatafeedResults, this._basicValidations, jobConfig, datafeedConfig);
    this._validationSummary.basic = this._isOverallBasicValid();
    // Update validation results subject
    this._basicValidationResult$.next(this._basicValidations);
  }
  _runAdvancedValidation() {
    if ((0, _job_creator.isCategorizationJobCreator)(this._jobCreator)) {
      this._advancedValidations.categorizationFieldValid.valid = this._jobCreator.overallValidStatus !== _categorization_job.CATEGORY_EXAMPLES_VALIDATION_STATUS.INVALID;
    }
  }
  _isOverallBasicValid() {
    return Object.values(this._basicValidations).some(v => v.valid === false) === false;
  }
  get validationSummary() {
    return this._validationSummary;
  }
  get bucketSpan() {
    return this._basicValidations.bucketSpan;
  }
  get summaryCountField() {
    return this._basicValidations.summaryCountField;
  }
  get duplicateDetectors() {
    return this._basicValidations.duplicateDetectors;
  }
  get jobId() {
    return this._basicValidations.jobId;
  }
  get groupIds() {
    return this._basicValidations.groupIds;
  }
  get modelMemoryLimit() {
    return this._basicValidations.modelMemoryLimit;
  }
  get query() {
    return this._basicValidations.query;
  }
  get queryDelay() {
    return this._basicValidations.queryDelay;
  }
  get frequency() {
    return this._basicValidations.frequency;
  }
  get scrollSize() {
    return this._basicValidations.scrollSize;
  }
  set advancedValid(valid) {
    this._validationSummary.advanced = valid;
  }
  get validating() {
    return this._validating;
  }
  get categorizationField() {
    return this._advancedValidations.categorizationFieldValid.valid;
  }
  set categorizationField(valid) {
    this._advancedValidations.categorizationFieldValid.valid = valid;
  }
  get categorizerMissingPerPartition() {
    return this._basicValidations.categorizerMissingPerPartition;
  }
  get categorizerVaryingPerPartitionField() {
    return this._basicValidations.categorizerVaryingPerPartitionField;
  }

  /**
   * Indicates if the Pick Fields step has a valid input
   */
  get isPickFieldsStepValid() {
    return this._jobCreator.detectors.length > 0 && (this._jobCreator.type !== _new_job.JOB_TYPE.ADVANCED || this._jobCreator.type === _new_job.JOB_TYPE.ADVANCED && this.modelMemoryLimit.valid) && this.bucketSpan.valid && this.duplicateDetectors.valid && this.categorizerMissingPerPartition.valid && this.categorizerVaryingPerPartitionField.valid && this.summaryCountField.valid && !this.validating && (this._jobCreator.type !== _new_job.JOB_TYPE.CATEGORIZATION || this._jobCreator.type === _new_job.JOB_TYPE.CATEGORIZATION && this.categorizationField);
  }
  get isModelMemoryEstimationPayloadValid() {
    return this._jobCreator.detectors.length > 0 && this.bucketSpan.valid && this.duplicateDetectors.valid && (this._jobCreator.type !== _new_job.JOB_TYPE.CATEGORIZATION || this._jobCreator.type === _new_job.JOB_TYPE.CATEGORIZATION && this.categorizationField);
  }
}
exports.JobValidator = JobValidator;