"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildConfig = buildConfig;
var _parse_interval = require("../../../../common/util/parse_interval");
var _anomaly_utils = require("../../../../common/util/anomaly_utils");
var _chart_config_builder = require("../../util/chart_config_builder");
var _job_service = require("../../services/job_service");
var _job_utils = require("../../../../common/util/job_utils");
var _aggregation_types = require("../../../../common/constants/aggregation_types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/*
 * Builds the configuration object used to plot a chart showing where the anomalies occur in
 * the raw data in the Explorer dashboard.
 */

// Builds the chart configuration for the provided anomaly record, returning
// an object with properties used for the display (series function and field, aggregation interval etc),
// and properties for the datafeed used for the job (indices, time field etc).
function buildConfig(record) {
  const job = _job_service.mlJobService.getJob(record.job_id);
  const detectorIndex = record.detector_index;
  const config = (0, _chart_config_builder.buildConfigFromDetector)(job, detectorIndex);

  // Add extra properties used by the explorer dashboard charts.
  config.functionDescription = record.function_description;
  config.bucketSpanSeconds = (0, _parse_interval.parseInterval)(job.analysis_config.bucket_span).asSeconds();
  config.detectorLabel = record.function;
  if (_job_service.mlJobService.detectorsByJob[record.job_id] !== undefined && detectorIndex < _job_service.mlJobService.detectorsByJob[record.job_id].length) {
    config.detectorLabel = _job_service.mlJobService.detectorsByJob[record.job_id][detectorIndex].detector_description;
  } else {
    if (record.field_name !== undefined) {
      config.detectorLabel += ` ${config.fieldName}`;
    }
  }
  if (record.field_name !== undefined) {
    config.fieldName = record.field_name;
    config.metricFieldName = record.field_name;
  }

  // Add the 'entity_fields' i.e. the partition, by, over fields which
  // define the metric series to be plotted.
  config.entityFields = (0, _anomaly_utils.getEntityFieldList)(record);
  if (record.function === _aggregation_types.ML_JOB_AGGREGATION.METRIC) {
    config.metricFunction = (0, _job_utils.mlFunctionToESAggregation)(record.function_description);
  }

  // Build the tooltip data for the chart info icon, showing further details on what is being plotted.
  let functionLabel = config.metricFunction;
  if (config.metricFieldName !== undefined) {
    functionLabel += ` ${config.metricFieldName}`;
  }
  config.infoTooltip = {
    jobId: record.job_id,
    aggregationInterval: config.interval,
    chartFunction: functionLabel,
    entityFields: config.entityFields.map(f => ({
      fieldName: f.fieldName,
      fieldValue: f.fieldValue
    }))
  };
  return config;
}