"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useCreateAnalyticsForm = void 0;
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _public = require("@kbn/data-plugin/public");
var _mlErrorUtils = require("@kbn/ml-error-utils");
var _ml_api_service = require("../../../../../services/ml_api_service");
var _ml = require("../../../../../contexts/ml");
var _common = require("../../../../common");
var _action_clone = require("../../components/action_clone");
var _actions = require("./actions");
var _reducer = require("./reducer");
var _state = require("./state");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function checkIndexExists(destinationIndex) {
  let resp;
  let errorMessage;
  try {
    resp = await _ml_api_service.ml.checkIndicesExists({
      indices: [destinationIndex]
    });
  } catch (e) {
    errorMessage = (0, _mlErrorUtils.extractErrorMessage)(e);
  }
  return {
    resp,
    errorMessage
  };
}
async function retryIndexExistsCheck(destinationIndex) {
  let retryCount = 15;
  let resp = await checkIndexExists(destinationIndex);
  let indexExists = resp.resp && resp.resp[destinationIndex] && resp.resp[destinationIndex].exists;
  while (retryCount > 1 && !indexExists) {
    retryCount--;
    await delay(1000);
    resp = await checkIndexExists(destinationIndex);
    indexExists = resp.resp && resp.resp[destinationIndex] && resp.resp[destinationIndex].exists;
  }
  if (indexExists) {
    return {
      success: true,
      indexExists: true
    };
  }
  return {
    success: false,
    indexExists: false,
    ...(resp.errorMessage !== undefined ? {
      errorMessage: resp.errorMessage
    } : {})
  };
}
function delay(ms = 1000) {
  return new Promise(resolve => {
    setTimeout(resolve, ms);
  });
}
const useCreateAnalyticsForm = () => {
  const mlContext = (0, _ml.useMlContext)();
  const [state, dispatch] = (0, _react.useReducer)(_reducer.reducer, (0, _state.getInitialState)());
  const {
    refresh
  } = (0, _common.useRefreshAnalyticsList)();
  const {
    form,
    jobConfig,
    isAdvancedEditorEnabled
  } = state;
  const {
    createIndexPattern,
    jobId
  } = form;
  let {
    destinationIndex
  } = form;
  const addRequestMessage = requestMessage => dispatch({
    type: _actions.ACTION.ADD_REQUEST_MESSAGE,
    requestMessage
  });
  const closeModal = () => dispatch({
    type: _actions.ACTION.CLOSE_MODAL
  });
  const resetAdvancedEditorMessages = () => dispatch({
    type: _actions.ACTION.RESET_ADVANCED_EDITOR_MESSAGES
  });
  const setAdvancedEditorRawString = advancedEditorRawString => dispatch({
    type: _actions.ACTION.SET_ADVANCED_EDITOR_RAW_STRING,
    advancedEditorRawString
  });
  const setIndexPatternTitles = payload => dispatch({
    type: _actions.ACTION.SET_INDEX_PATTERN_TITLES,
    payload
  });
  const setIsJobCreated = isJobCreated => dispatch({
    type: _actions.ACTION.SET_IS_JOB_CREATED,
    isJobCreated
  });
  const setIsJobStarted = isJobStarted => {
    dispatch({
      type: _actions.ACTION.SET_IS_JOB_STARTED,
      isJobStarted
    });
  };
  const resetRequestMessages = () => dispatch({
    type: _actions.ACTION.RESET_REQUEST_MESSAGES
  });
  const resetForm = () => dispatch({
    type: _actions.ACTION.RESET_FORM
  });
  const createAnalyticsJob = async () => {
    resetRequestMessages();
    const analyticsJobConfig = isAdvancedEditorEnabled ? jobConfig : (0, _state.getJobConfigFromFormState)(form);
    if (isAdvancedEditorEnabled) {
      destinationIndex = analyticsJobConfig.dest.index;
    }
    try {
      await _ml_api_service.ml.dataFrameAnalytics.createDataFrameAnalytics(jobId, analyticsJobConfig);
      addRequestMessage({
        message: _i18n.i18n.translate('xpack.ml.dataframe.stepCreateForm.createDataFrameAnalyticsSuccessMessage', {
          defaultMessage: 'Request to create data frame analytics {jobId} acknowledged.',
          values: {
            jobId
          }
        })
      });
      setIsJobCreated(true);
      if (createIndexPattern) {
        createKibanaIndexPattern();
      }
      refresh();
      return true;
    } catch (e) {
      addRequestMessage({
        error: (0, _mlErrorUtils.extractErrorMessage)(e),
        message: _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.errorCreatingDataFrameAnalyticsJob', {
          defaultMessage: 'An error occurred creating the data frame analytics job:'
        })
      });
      return false;
    }
  };
  const createKibanaIndexPattern = async () => {
    const dataViewName = destinationIndex;
    const exists = await retryIndexExistsCheck(destinationIndex);
    if ((exists === null || exists === void 0 ? void 0 : exists.success) === true) {
      // index exists - create data view
      if ((exists === null || exists === void 0 ? void 0 : exists.indexExists) === true) {
        try {
          await mlContext.dataViewsContract.createAndSave({
            title: dataViewName,
            ...(form.timeFieldName ? {
              timeFieldName: form.timeFieldName
            } : {})
          }, false, true);
          addRequestMessage({
            message: _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.createDataViewSuccessMessage', {
              defaultMessage: 'Kibana data view {dataViewName} created.',
              values: {
                dataViewName
              }
            })
          });
        } catch (e) {
          // handle data view creation error
          if (e instanceof _public.DuplicateDataViewError) {
            addRequestMessage({
              error: _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.duplicateDataViewErrorMessageError', {
                defaultMessage: 'The data view {dataViewName} already exists.',
                values: {
                  dataViewName
                }
              }),
              message: _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.duplicateDataViewErrorMessage', {
                defaultMessage: 'An error occurred creating the Kibana data view:'
              })
            });
          } else {
            addRequestMessage({
              error: (0, _mlErrorUtils.extractErrorMessage)(e),
              message: _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.createDataViewErrorMessage', {
                defaultMessage: 'An error occurred creating the Kibana data view:'
              })
            });
          }
        }
      }
    } else {
      // Ran out of retries or there was a problem checking index exists
      if (exists !== null && exists !== void 0 && exists.errorMessage) {
        addRequestMessage({
          error: _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.errorCheckingDestinationIndexDataFrameAnalyticsJob', {
            defaultMessage: '{errorMessage}',
            values: {
              errorMessage: exists.errorMessage
            }
          }),
          message: _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.errorOccurredCheckingDestinationIndexDataFrameAnalyticsJob', {
            defaultMessage: 'An error occurred checking destination index exists.'
          })
        });
      } else {
        addRequestMessage({
          error: _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.destinationIndexNotCreatedForDataFrameAnalyticsJob', {
            defaultMessage: 'Destination index has not yet been created.'
          }),
          message: _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.unableToCreateDataViewForDataFrameAnalyticsJob', {
            defaultMessage: 'Unable to create data view.'
          })
        });
      }
    }
  };
  const prepareFormValidation = async () => {
    try {
      // Set the existing data view names.
      const indexPatternsMap = {};
      const savedObjects = (await mlContext.dataViewsContract.getCache()) || [];
      savedObjects.forEach(obj => {
        var _obj$attributes;
        const title = obj === null || obj === void 0 ? void 0 : (_obj$attributes = obj.attributes) === null || _obj$attributes === void 0 ? void 0 : _obj$attributes.title;
        if (title !== undefined) {
          const id = (obj === null || obj === void 0 ? void 0 : obj.id) || '';
          indexPatternsMap[title] = {
            label: title,
            value: id
          };
        }
      });
      setIndexPatternTitles({
        indexPatternsMap
      });
    } catch (e) {
      addRequestMessage({
        error: (0, _mlErrorUtils.extractErrorMessage)(e),
        message: _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.errorGettingDataViewNames', {
          defaultMessage: 'An error occurred getting the existing data view names:'
        })
      });
    }
  };
  const initiateWizard = async () => {
    await mlContext.dataViewsContract.clearCache();
    await prepareFormValidation();
  };
  const startAnalyticsJob = async () => {
    try {
      const response = await _ml_api_service.ml.dataFrameAnalytics.startDataFrameAnalytics(jobId);
      if (response.acknowledged !== true) {
        throw new Error(response);
      }
      addRequestMessage({
        message: _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.startDataFrameAnalyticsSuccessMessage', {
          defaultMessage: 'Request to start data frame analytics {jobId} acknowledged.',
          values: {
            jobId
          }
        })
      });
      setIsJobStarted(true);
      refresh();
    } catch (e) {
      addRequestMessage({
        error: (0, _mlErrorUtils.extractErrorMessage)(e),
        message: _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.errorStartingDataFrameAnalyticsJob', {
          defaultMessage: 'An error occurred starting the data frame analytics job:'
        })
      });
    }
  };
  const setJobConfig = payload => {
    dispatch({
      type: _actions.ACTION.SET_JOB_CONFIG,
      payload
    });
  };
  const setFormState = payload => {
    dispatch({
      type: _actions.ACTION.SET_FORM_STATE,
      payload
    });
  };
  const switchToAdvancedEditor = () => {
    dispatch({
      type: _actions.ACTION.SWITCH_TO_ADVANCED_EDITOR
    });
  };
  const switchToForm = () => {
    dispatch({
      type: _actions.ACTION.SWITCH_TO_FORM
    });
  };
  const setEstimatedModelMemoryLimit = value => {
    dispatch({
      type: _actions.ACTION.SET_ESTIMATED_MODEL_MEMORY_LIMIT,
      value: value !== null && value !== void 0 ? value : ''
    });
  };
  const setJobClone = async cloneJob => {
    resetForm();
    const config = (0, _action_clone.extractCloningConfig)(cloneJob);
    if ((0, _action_clone.isAdvancedConfig)(config)) {
      setFormState((0, _state.getFormStateFromJobConfig)(config));
      switchToAdvancedEditor();
    } else {
      setFormState((0, _state.getFormStateFromJobConfig)(config));
      setEstimatedModelMemoryLimit(config.model_memory_limit);
    }
    dispatch({
      type: _actions.ACTION.SET_JOB_CLONE,
      cloneJob
    });
  };
  const actions = {
    closeModal,
    createAnalyticsJob,
    initiateWizard,
    resetAdvancedEditorMessages,
    setAdvancedEditorRawString,
    setFormState,
    setJobConfig,
    startAnalyticsJob,
    switchToAdvancedEditor,
    switchToForm,
    setEstimatedModelMemoryLimit,
    setJobClone
  };
  return {
    state,
    actions
  };
};
exports.useCreateAnalyticsForm = useCreateAnalyticsForm;