"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getModelMemoryLimitErrors = getModelMemoryLimitErrors;
exports.mmlUnitInvalidErrorMessage = void 0;
exports.reducer = reducer;
exports.validateAdvancedEditor = void 0;
exports.validateMinMML = validateMinMML;
exports.validateNumTopFeatureImportanceValues = void 0;
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _numeral = _interopRequireDefault(require("@elastic/numeral"));
var _public = require("@kbn/data-plugin/public");
var _public2 = require("@kbn/es-ui-shared-plugin/public");
var _es_utils = require("../../../../../../../common/util/es_utils");
var _actions = require("./actions");
var _state = require("./state");
var _job_utils = require("../../../../../../../common/util/job_utils");
var _validators = require("../../../../../../../common/util/validators");
var _validation = require("../../../../../../../common/constants/validation");
var _data_frame_analytics = require("../../../../../../../common/constants/data_frame_analytics");
var _analytics = require("../../../../common/analytics");
var _clone_action_name = require("../../components/action_clone/clone_action_name");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// @ts-ignore

const {
  collapseLiteralStrings
} = _public2.XJson;
const mmlAllowedUnitsStr = `${_validation.ALLOWED_DATA_UNITS.slice(0, _validation.ALLOWED_DATA_UNITS.length - 1).join(', ')} or ${[..._validation.ALLOWED_DATA_UNITS].pop()}`;
const mmlUnitInvalidErrorMessage = _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.modelMemoryUnitsInvalidError', {
  defaultMessage: 'Model memory limit data unit unrecognized. It must be {str}',
  values: {
    str: mmlAllowedUnitsStr
  }
});

/**
 * Returns the list of model memory limit errors based on validation result.
 * @param mmlValidationResult
 */
exports.mmlUnitInvalidErrorMessage = mmlUnitInvalidErrorMessage;
function getModelMemoryLimitErrors(mmlValidationResult) {
  if (mmlValidationResult === null) {
    return null;
  }
  return Object.keys(mmlValidationResult).reduce((acc, errorKey) => {
    if (errorKey === 'min') {
      acc.push(_i18n.i18n.translate('xpack.ml.dataframe.analytics.create.modelMemoryUnitsMinError', {
        defaultMessage: 'Model memory limit is lower than estimated value {mml}',
        values: {
          mml: mmlValidationResult.min.minValue
        }
      }));
    }
    if (errorKey === 'invalidUnits') {
      acc.push(_i18n.i18n.translate('xpack.ml.dataframe.analytics.create.modelMemoryUnitsInvalidError', {
        defaultMessage: 'Model memory limit data unit unrecognized. It must be {str}',
        values: {
          str: mmlAllowedUnitsStr
        }
      }));
    }
    return acc;
  }, []);
}
const getSourceIndexString = state => {
  var _jobConfig$source;
  const {
    jobConfig
  } = state;
  const sourceIndex = jobConfig === null || jobConfig === void 0 ? void 0 : (_jobConfig$source = jobConfig.source) === null || _jobConfig$source === void 0 ? void 0 : _jobConfig$source.index;
  if (typeof sourceIndex === 'string') {
    return sourceIndex;
  }
  if (Array.isArray(sourceIndex)) {
    return sourceIndex.join(',');
  }
  return '';
};
const isSourceIndexNameValid = (sourceIndexName, sourceIndex) => {
  // general check against Kibana data view names, but since this is about the advanced editor
  // with support for arrays in the job config, we also need to check that each individual name
  // doesn't include a comma if index names are supplied as an array.
  // `indexPatterns.validate()` returns a map of messages, we're only interested here if it's valid or not.
  // If there are no messages, it means the source index name is valid.
  let sourceIndexNameValid = Object.keys(_public.indexPatterns.validate(sourceIndexName)).length === 0;
  if (sourceIndexNameValid) {
    if (typeof sourceIndex === 'string') {
      sourceIndexNameValid = !sourceIndex.includes(',');
    }
    if (Array.isArray(sourceIndex)) {
      sourceIndexNameValid = !sourceIndex.some(d => d === null || d === void 0 ? void 0 : d.includes(','));
    }
  }
  return sourceIndexNameValid;
};

/**
 * Validates num_top_feature_importance_values. Must be an integer >= 0.
 */
const validateNumTopFeatureImportanceValues = numTopFeatureImportanceValues => {
  return typeof numTopFeatureImportanceValues === 'number' && numTopFeatureImportanceValues >= _analytics.NUM_TOP_FEATURE_IMPORTANCE_VALUES_MIN && Number.isInteger(numTopFeatureImportanceValues);
};
exports.validateNumTopFeatureImportanceValues = validateNumTopFeatureImportanceValues;
const validateAdvancedEditor = state => {
  var _jobConfig$source2, _jobConfig$dest$index, _jobConfig$dest, _jobConfig$analyzed_f, _jobConfig$dest2, _jobConfig$dest3;
  const {
    jobIdEmpty,
    jobIdValid,
    jobIdExists,
    jobType,
    createIndexPattern
  } = state.form;
  const {
    jobConfig
  } = state;
  state.advancedEditorMessages = [];
  const sourceIndexName = getSourceIndexString(state);
  const sourceIndexNameEmpty = sourceIndexName === '';
  const sourceIndex = jobConfig === null || jobConfig === void 0 ? void 0 : (_jobConfig$source2 = jobConfig.source) === null || _jobConfig$source2 === void 0 ? void 0 : _jobConfig$source2.index;
  const sourceIndexNameValid = isSourceIndexNameValid(sourceIndexName, sourceIndex);
  const destinationIndexName = (_jobConfig$dest$index = jobConfig === null || jobConfig === void 0 ? void 0 : (_jobConfig$dest = jobConfig.dest) === null || _jobConfig$dest === void 0 ? void 0 : _jobConfig$dest.index) !== null && _jobConfig$dest$index !== void 0 ? _jobConfig$dest$index : '';
  const destinationIndexNameEmpty = destinationIndexName === '';
  const destinationIndexNameValid = (0, _es_utils.isValidIndexName)(destinationIndexName);
  const destinationIndexPatternTitleExists = state.indexPatternsMap[destinationIndexName] !== undefined;
  const analyzedFields = (jobConfig === null || jobConfig === void 0 ? void 0 : (_jobConfig$analyzed_f = jobConfig.analyzed_fields) === null || _jobConfig$analyzed_f === void 0 ? void 0 : _jobConfig$analyzed_f.includes) || [];
  const resultsFieldEmptyString = typeof (jobConfig === null || jobConfig === void 0 ? void 0 : (_jobConfig$dest2 = jobConfig.dest) === null || _jobConfig$dest2 === void 0 ? void 0 : _jobConfig$dest2.results_field) === 'string' && (jobConfig === null || jobConfig === void 0 ? void 0 : (_jobConfig$dest3 = jobConfig.dest) === null || _jobConfig$dest3 === void 0 ? void 0 : _jobConfig$dest3.results_field.trim()) === '';
  const mml = jobConfig.model_memory_limit;
  const modelMemoryLimitEmpty = mml === '' || mml === undefined;
  if (!modelMemoryLimitEmpty && mml !== undefined) {
    const {
      valid
    } = (0, _job_utils.validateModelMemoryLimitUnits)(mml);
    state.form.modelMemoryLimitUnitValid = valid;
  }
  let dependentVariableEmpty = false;
  let includesValid = true;
  let trainingPercentValid = true;
  let numTopFeatureImportanceValuesValid = true;
  if (jobConfig.analysis === undefined && (jobType === _data_frame_analytics.ANALYSIS_CONFIG_TYPE.CLASSIFICATION || jobType === _data_frame_analytics.ANALYSIS_CONFIG_TYPE.REGRESSION)) {
    dependentVariableEmpty = true;
  }
  if (jobConfig.analysis !== undefined && ((0, _analytics.isRegressionAnalysis)(jobConfig.analysis) || (0, _analytics.isClassificationAnalysis)(jobConfig.analysis))) {
    const dependentVariableName = (0, _analytics.getDependentVar)(jobConfig.analysis) || '';
    dependentVariableEmpty = dependentVariableName === '';
    if (!dependentVariableEmpty && Array.isArray(analyzedFields) && analyzedFields.length > 0 && !analyzedFields.includes(dependentVariableName)) {
      includesValid = false;
      state.advancedEditorMessages.push({
        error: _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.advancedEditorMessage.includesInvalid', {
          defaultMessage: 'The dependent variable must be included.'
        }),
        message: ''
      });
    }
    const trainingPercent = (0, _analytics.getTrainingPercent)(jobConfig.analysis);
    if (trainingPercent !== undefined && (isNaN(trainingPercent) || typeof trainingPercent !== 'number' || trainingPercent < _analytics.TRAINING_PERCENT_MIN || trainingPercent > _analytics.TRAINING_PERCENT_MAX)) {
      trainingPercentValid = false;
      state.advancedEditorMessages.push({
        error: _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.advancedEditorMessage.trainingPercentInvalid', {
          defaultMessage: 'The training percent must be a number between {min} and {max}.',
          values: {
            min: _analytics.TRAINING_PERCENT_MIN,
            max: _analytics.TRAINING_PERCENT_MAX
          }
        }),
        message: ''
      });
    }
    const numTopFeatureImportanceValues = (0, _analytics.getNumTopFeatureImportanceValues)(jobConfig.analysis);
    if (numTopFeatureImportanceValues !== undefined) {
      numTopFeatureImportanceValuesValid = validateNumTopFeatureImportanceValues(numTopFeatureImportanceValues);
      if (numTopFeatureImportanceValuesValid === false) {
        state.advancedEditorMessages.push({
          error: _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.advancedEditorMessage.numTopFeatureImportanceValuesInvalid', {
            defaultMessage: 'The value for num_top_feature_importance_values must be an integer of {min} or higher.',
            values: {
              min: 0
            }
          }),
          message: ''
        });
      }
    }
  }
  if (sourceIndexNameEmpty) {
    state.advancedEditorMessages.push({
      error: _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.advancedEditorMessage.sourceIndexNameEmpty', {
        defaultMessage: 'The source index name must not be empty.'
      }),
      message: ''
    });
  } else if (!sourceIndexNameValid) {
    state.advancedEditorMessages.push({
      error: _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.advancedEditorMessage.sourceIndexNameValid', {
        defaultMessage: 'Invalid source index name.'
      }),
      message: ''
    });
  }
  if (destinationIndexNameEmpty) {
    state.advancedEditorMessages.push({
      error: _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.advancedEditorMessage.destinationIndexNameEmpty', {
        defaultMessage: 'The destination index name must not be empty.'
      }),
      message: ''
    });
  } else if (destinationIndexPatternTitleExists && !createIndexPattern) {
    state.advancedEditorMessages.push({
      error: _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.advancedEditorMessage.destinationIndexNameExistsWarn', {
        defaultMessage: 'An index with this destination index name already exists. Be aware that running this analytics job will modify this destination index.'
      }),
      message: ''
    });
  } else if (!destinationIndexNameValid) {
    state.advancedEditorMessages.push({
      error: _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.advancedEditorMessage.destinationIndexNameValid', {
        defaultMessage: 'Invalid destination index name.'
      }),
      message: ''
    });
  }
  if (resultsFieldEmptyString) {
    state.advancedEditorMessages.push({
      error: _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.advancedEditorMessage.resultsFieldEmptyString', {
        defaultMessage: 'The results field must not be an empty string.'
      }),
      message: ''
    });
  }
  if (dependentVariableEmpty) {
    state.advancedEditorMessages.push({
      error: _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.advancedEditorMessage.dependentVariableEmpty', {
        defaultMessage: 'The dependent variable field must not be empty.'
      }),
      message: ''
    });
  }
  if (modelMemoryLimitEmpty) {
    state.advancedEditorMessages.push({
      error: _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.advancedEditorMessage.modelMemoryLimitEmpty', {
        defaultMessage: 'The model memory limit field must not be empty.'
      }),
      message: ''
    });
  }
  if (!state.form.modelMemoryLimitUnitValid) {
    state.advancedEditorMessages.push({
      error: mmlUnitInvalidErrorMessage,
      message: ''
    });
  }
  state.form.destinationIndexPatternTitleExists = destinationIndexPatternTitleExists;
  state.isValid = includesValid && trainingPercentValid && state.form.modelMemoryLimitUnitValid && !jobIdEmpty && jobIdValid && !jobIdExists && !sourceIndexNameEmpty && sourceIndexNameValid && !destinationIndexNameEmpty && destinationIndexNameValid && !resultsFieldEmptyString && !dependentVariableEmpty && !modelMemoryLimitEmpty && numTopFeatureImportanceValuesValid && (!destinationIndexPatternTitleExists || !createIndexPattern);
  return state;
};

/**
 * Validates provided MML isn't lower than the estimated one.
 */
exports.validateAdvancedEditor = validateAdvancedEditor;
function validateMinMML(estimatedMml) {
  return mml => {
    if (!mml || !estimatedMml) {
      return null;
    }

    // @ts-ignore
    const mmlInBytes = (0, _numeral.default)(mml.toUpperCase()).value();
    // @ts-ignore
    const estimatedMmlInBytes = (0, _numeral.default)(estimatedMml.toUpperCase()).value();
    return estimatedMmlInBytes > mmlInBytes ? {
      min: {
        minValue: estimatedMml,
        actualValue: mml
      }
    } : null;
  };
}

/**
 * Result validator function for the MML.
 * Re-init only if the estimated mml has been changed.
 */
const mmlValidator = (0, _lodash.memoize)(estimatedMml => (0, _validators.composeValidators)((0, _validators.requiredValidator)(), validateMinMML(estimatedMml), (0, _validators.memoryInputValidator)()));
const validateMml = (0, _lodash.memoize)((estimatedMml, mml) => mmlValidator(estimatedMml)(mml), (...args) => args.join('_'));
const validateForm = state => {
  const {
    jobIdEmpty,
    jobIdValid,
    jobIdExists,
    jobType,
    sourceIndexNameEmpty,
    sourceIndexNameValid,
    destinationIndexNameEmpty,
    destinationIndexNameValid,
    destinationIndexPatternTitleExists,
    createIndexPattern,
    dependentVariable,
    modelMemoryLimit,
    numTopFeatureImportanceValuesValid
  } = state.form;
  const {
    estimatedModelMemoryLimit
  } = state;
  const jobTypeEmpty = jobType === undefined;
  const dependentVariableEmpty = (jobType === _data_frame_analytics.ANALYSIS_CONFIG_TYPE.REGRESSION || jobType === _data_frame_analytics.ANALYSIS_CONFIG_TYPE.CLASSIFICATION) && dependentVariable === '';
  const mmlValidationResult = validateMml(estimatedModelMemoryLimit, modelMemoryLimit);
  const mmlInvalid = mmlValidationResult !== null && (mmlValidationResult.invalidUnits !== undefined || mmlValidationResult.required === true);
  state.form.modelMemoryLimitValidationResult = mmlValidationResult;
  state.isValid = !jobTypeEmpty && !mmlInvalid && !jobIdEmpty && jobIdValid && !jobIdExists && !sourceIndexNameEmpty && sourceIndexNameValid && !destinationIndexNameEmpty && destinationIndexNameValid && !dependentVariableEmpty && numTopFeatureImportanceValuesValid && (!destinationIndexPatternTitleExists || !createIndexPattern);
  return state;
};
function reducer(state, action) {
  var _config$source;
  switch (action.type) {
    case _actions.ACTION.ADD_REQUEST_MESSAGE:
      const requestMessages = state.requestMessages;
      requestMessages.push(action.requestMessage);
      return {
        ...state,
        requestMessages
      };
    case _actions.ACTION.RESET_REQUEST_MESSAGES:
      return {
        ...state,
        requestMessages: []
      };
    case _actions.ACTION.RESET_ADVANCED_EDITOR_MESSAGES:
      return {
        ...state,
        advancedEditorMessages: []
      };
    case _actions.ACTION.RESET_FORM:
      return (0, _state.getInitialState)();
    case _actions.ACTION.SET_ADVANCED_EDITOR_RAW_STRING:
      let resultJobConfig;
      let disableSwitchToForm = false;
      try {
        resultJobConfig = JSON.parse(collapseLiteralStrings(action.advancedEditorRawString));
        const runtimeMappingsChanged = state.form.runtimeMappings && resultJobConfig.source.runtime_mappings && !(0, _lodash.isEqual)(state.form.runtimeMappings, resultJobConfig.source.runtime_mappings);
        disableSwitchToForm = (0, _clone_action_name.isAdvancedConfig)(resultJobConfig) || runtimeMappingsChanged;
      } catch (e) {
        return {
          ...state,
          advancedEditorRawString: action.advancedEditorRawString,
          isAdvancedEditorValidJson: false,
          disableSwitchToForm: true,
          advancedEditorMessages: []
        };
      }
      return {
        ...validateAdvancedEditor({
          ...state,
          jobConfig: resultJobConfig
        }),
        advancedEditorRawString: action.advancedEditorRawString,
        isAdvancedEditorValidJson: true,
        disableSwitchToForm
      };
    case _actions.ACTION.SET_FORM_STATE:
      const newFormState = {
        ...state.form,
        ...action.payload
      };

      // update state attributes which are derived from other state attributes.
      if (action.payload.destinationIndex !== undefined) {
        newFormState.destinationIndexNameEmpty = newFormState.destinationIndex === '';
        newFormState.destinationIndexNameValid = (0, _es_utils.isValidIndexName)(newFormState.destinationIndex);
        newFormState.destinationIndexPatternTitleExists = state.indexPatternsMap[newFormState.destinationIndex] !== undefined;
      }
      if (action.payload.jobId !== undefined) {
        newFormState.jobIdEmpty = newFormState.jobId === '';
        newFormState.jobIdValid = (0, _job_utils.isJobIdValid)(newFormState.jobId);
        newFormState.jobIdInvalidMaxLength = !!(0, _validators.maxLengthValidator)(_validation.JOB_ID_MAX_LENGTH)(newFormState.jobId);
      }
      if (action.payload.sourceIndex !== undefined) {
        newFormState.sourceIndexNameEmpty = newFormState.sourceIndex === '';
        const validationMessages = _public.indexPatterns.validate(newFormState.sourceIndex);
        newFormState.sourceIndexNameValid = Object.keys(validationMessages).length === 0;
      }
      if (action.payload.numTopFeatureImportanceValues !== undefined) {
        newFormState.numTopFeatureImportanceValuesValid = validateNumTopFeatureImportanceValues(newFormState === null || newFormState === void 0 ? void 0 : newFormState.numTopFeatureImportanceValues);
      }
      return state.isAdvancedEditorEnabled ? validateAdvancedEditor({
        ...state,
        form: newFormState
      }) : validateForm({
        ...state,
        form: newFormState
      });
    case _actions.ACTION.SET_INDEX_PATTERN_TITLES:
      {
        const newState = {
          ...state,
          ...action.payload
        };
        newState.form.destinationIndexPatternTitleExists = newState.indexPatternsMap[newState.form.destinationIndex] !== undefined;
        return newState;
      }
    case _actions.ACTION.SET_IS_JOB_CREATED:
      return {
        ...state,
        isJobCreated: action.isJobCreated
      };
    case _actions.ACTION.SET_IS_JOB_STARTED:
      return {
        ...state,
        isJobStarted: action.isJobStarted
      };
    case _actions.ACTION.SET_JOB_CONFIG:
      return validateAdvancedEditor({
        ...state,
        jobConfig: action.payload
      });
    case _actions.ACTION.SWITCH_TO_ADVANCED_EDITOR:
      const jobConfig = (0, _state.getJobConfigFromFormState)(state.form);
      const shouldDisableSwitchToForm = (0, _clone_action_name.isAdvancedConfig)(jobConfig);
      return validateAdvancedEditor({
        ...state,
        advancedEditorRawString: JSON.stringify(jobConfig, null, 2),
        isAdvancedEditorEnabled: true,
        disableSwitchToForm: shouldDisableSwitchToForm,
        hasSwitchedToEditor: true,
        jobConfig
      });
    case _actions.ACTION.SWITCH_TO_FORM:
      const {
        jobConfig: config
      } = state;
      const {
        jobId
      } = state.form;
      // @ts-ignore
      const formState = (0, _state.getFormStateFromJobConfig)(config, false);
      if (typeof jobId === 'string' && jobId.trim() !== '') {
        formState.jobId = jobId;
      }
      formState.jobIdEmpty = jobId === '';
      formState.jobIdValid = (0, _job_utils.isJobIdValid)(jobId);
      formState.jobIdInvalidMaxLength = !!(0, _validators.maxLengthValidator)(_validation.JOB_ID_MAX_LENGTH)(jobId);
      formState.destinationIndexNameEmpty = formState.destinationIndex === '';
      formState.destinationIndexNameValid = (0, _es_utils.isValidIndexName)(formState.destinationIndex || '');
      formState.destinationIndexPatternTitleExists = state.indexPatternsMap[formState.destinationIndex || ''] !== undefined;
      if (formState.numTopFeatureImportanceValues !== undefined) {
        formState.numTopFeatureImportanceValuesValid = validateNumTopFeatureImportanceValues(formState.numTopFeatureImportanceValues);
      }
      const sourceIndexName = getSourceIndexString(state);
      const sourceIndex = config === null || config === void 0 ? void 0 : (_config$source = config.source) === null || _config$source === void 0 ? void 0 : _config$source.index;
      const sourceIndexNameValid = isSourceIndexNameValid(sourceIndexName, sourceIndex);
      formState.sourceIndexNameValid = sourceIndexNameValid;
      return validateForm({
        ...state,
        // @ts-ignore
        form: formState,
        isAdvancedEditorEnabled: false,
        advancedEditorRawString: JSON.stringify(config, null, 2),
        jobConfig: config
      });
    case _actions.ACTION.SET_ESTIMATED_MODEL_MEMORY_LIMIT:
      return {
        ...state,
        estimatedModelMemoryLimit: action.value
      };
    case _actions.ACTION.SET_JOB_CLONE:
      return {
        ...state,
        cloneJob: action.cloneJob
      };
  }
  return state;
}