"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useIndexData = void 0;
var _react = require("react");
var _mlIsPopulatedObject = require("@kbn/ml-is-populated-object");
var _mlErrorUtils = require("@kbn/ml-error-utils");
var _runtime_field_utils = require("../../../../../../common/util/runtime_field_utils");
var _field_histograms = require("../../../../../../common/constants/field_histograms");
var _data_loader = require("../../../../datavisualizer/index_based/data_loader");
var _data_grid = require("../../../../components/data_grid");
var _analytics = require("../../../common/analytics");
var _ml_api_service = require("../../../../services/ml_api_service");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getRuntimeFieldColumns(runtimeMappings) {
  return Object.keys(runtimeMappings).map(id => {
    let field = runtimeMappings[id];
    if (Array.isArray(field)) {
      field = field[0];
    }
    const schema = (0, _data_grid.getDataGridSchemaFromESFieldType)(field.type);
    return {
      id,
      schema,
      isExpandable: schema !== 'boolean',
      isRuntimeFieldColumn: true
    };
  });
}
function getIndexPatternColumns(indexPattern, fieldsFilter) {
  const {
    fields
  } = indexPattern;
  return fields.filter(field => fieldsFilter.includes(field.name)).map(field => {
    const schema =
    // @ts-expect-error field is not DataViewField
    (0, _data_grid.getDataGridSchemaFromESFieldType)(field.type) || (0, _data_grid.getDataGridSchemaFromKibanaFieldType)(field);
    return {
      id: field.name,
      schema,
      isExpandable: schema !== 'boolean',
      isRuntimeFieldColumn: false
    };
  });
}
const useIndexData = (indexPattern, query, toastNotifications, runtimeMappings) => {
  // Fetch 500 random documents to determine populated fields.
  // This is a workaround to avoid passing potentially thousands of unpopulated fields
  // (for example, as part of filebeat/metricbeat/ECS based indices)
  // to the data grid component which would significantly slow down the page.
  const [indexPatternFields, setIndexPatternFields] = (0, _react.useState)();
  const [timeRangeMs, setTimeRangeMs] = (0, _react.useState)();
  (0, _react.useEffect)(() => {
    async function fetchDataGridSampleDocuments() {
      setErrorMessage('');
      setStatus(_analytics.INDEX_STATUS.LOADING);
      const esSearchRequest = {
        index: indexPattern.title,
        body: {
          fields: ['*'],
          _source: false,
          query: {
            function_score: {
              query: {
                match_all: {}
              },
              random_score: {}
            }
          },
          size: 500
        }
      };
      try {
        const resp = await _ml_api_service.ml.esSearch(esSearchRequest);
        const docs = resp.hits.hits.map(d => {
          var _d$fields;
          return (0, _data_grid.getProcessedFields)((_d$fields = d.fields) !== null && _d$fields !== void 0 ? _d$fields : {});
        });

        // Get all field names for each returned doc and flatten it
        // to a list of unique field names used across all docs.
        const allDataViewFields = (0, _data_grid.getFieldsFromKibanaIndexPattern)(indexPattern);
        const populatedFields = [...new Set(docs.map(Object.keys).flat(1))].filter(d => allDataViewFields.includes(d)).sort();
        setStatus(_analytics.INDEX_STATUS.LOADED);
        setIndexPatternFields(populatedFields);
      } catch (e) {
        setErrorMessage((0, _mlErrorUtils.extractErrorMessage)(e));
        setStatus(_analytics.INDEX_STATUS.ERROR);
      }
    }
    fetchDataGridSampleDocuments();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  // To be used for data grid column selection
  // and will be applied to doc and chart queries.
  const combinedRuntimeMappings = (0, _react.useMemo)(() => (0, _data_grid.getCombinedRuntimeMappings)(indexPattern, runtimeMappings), [indexPattern, runtimeMappings]);

  // Available data grid columns, will be a combination of index pattern and runtime fields.
  const [columns, setColumns] = (0, _react.useState)([]);
  (0, _react.useEffect)(() => {
    if (Array.isArray(indexPatternFields)) {
      setColumns([...getIndexPatternColumns(indexPattern, indexPatternFields), ...(combinedRuntimeMappings ? getRuntimeFieldColumns(combinedRuntimeMappings) : [])]);
    }
  }, [indexPattern, indexPatternFields, combinedRuntimeMappings]);
  const dataGrid = (0, _data_grid.useDataGrid)(columns);
  const {
    pagination,
    resetPagination,
    setErrorMessage,
    setRowCountInfo,
    setStatus,
    setTableItems,
    sortingColumns,
    tableItems
  } = dataGrid;
  (0, _react.useEffect)(() => {
    resetPagination();
    // custom comparison
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [JSON.stringify(query)]);
  (0, _react.useEffect)(() => {
    async function fetchIndexData() {
      var _indexPattern$getTime;
      setErrorMessage('');
      setStatus(_analytics.INDEX_STATUS.LOADING);
      const timeFieldName = (_indexPattern$getTime = indexPattern.getTimeField()) === null || _indexPattern$getTime === void 0 ? void 0 : _indexPattern$getTime.name;
      const sort = sortingColumns.reduce((s, column) => {
        s[column.id] = {
          order: column.direction
        };
        return s;
      }, {});
      const esSearchRequest = {
        index: indexPattern.title,
        body: {
          query,
          from: pagination.pageIndex * pagination.pageSize,
          size: pagination.pageSize,
          fields: [...(indexPatternFields !== null && indexPatternFields !== void 0 ? indexPatternFields : []), ...((0, _runtime_field_utils.isRuntimeMappings)(combinedRuntimeMappings) ? Object.keys(combinedRuntimeMappings) : [])],
          _source: false,
          ...(Object.keys(sort).length > 0 ? {
            sort
          } : {}),
          ...((0, _runtime_field_utils.isRuntimeMappings)(combinedRuntimeMappings) ? {
            runtime_mappings: combinedRuntimeMappings
          } : {}),
          ...(timeFieldName ? {
            aggs: {
              earliest: {
                min: {
                  field: timeFieldName
                }
              },
              latest: {
                max: {
                  field: timeFieldName
                }
              }
            }
          } : {})
        }
      };
      try {
        const resp = await _ml_api_service.ml.esSearch(esSearchRequest);
        if (resp.aggregations && (0, _mlIsPopulatedObject.isPopulatedObject)(resp.aggregations.earliest, ['value']) && (0, _mlIsPopulatedObject.isPopulatedObject)(resp.aggregations.latest, ['value'])) {
          setTimeRangeMs({
            from: resp.aggregations.earliest.value,
            to: resp.aggregations.latest.value
          });
        }
        const docs = resp.hits.hits.map(d => {
          var _d$fields2;
          return (0, _data_grid.getProcessedFields)((_d$fields2 = d.fields) !== null && _d$fields2 !== void 0 ? _d$fields2 : {});
        });
        setRowCountInfo({
          rowCount: typeof resp.hits.total === 'number' ? resp.hits.total : resp.hits.total.value,
          rowCountRelation: typeof resp.hits.total === 'number' ? 'eq' : resp.hits.total.relation
        });
        setTableItems(docs);
        setStatus(_analytics.INDEX_STATUS.LOADED);
      } catch (e) {
        setErrorMessage((0, _mlErrorUtils.extractErrorMessage)(e));
        setStatus(_analytics.INDEX_STATUS.ERROR);
      }
    }
    if (indexPatternFields !== undefined && query !== undefined) {
      fetchIndexData();
    }
    // custom comparison
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [indexPattern.title, indexPatternFields,
  // eslint-disable-next-line react-hooks/exhaustive-deps
  JSON.stringify([query, pagination, sortingColumns, combinedRuntimeMappings])]);
  const dataLoader = (0, _react.useMemo)(() => new _data_loader.DataLoader(indexPattern, toastNotifications),
  // eslint-disable-next-line react-hooks/exhaustive-deps
  [indexPattern]);
  (0, _react.useEffect)(() => {
    async function fetchColumnChartsData(fieldHistogramsQuery) {
      try {
        const columnChartsData = await dataLoader.loadFieldHistograms(columns.filter(cT => dataGrid.visibleColumns.includes(cT.id)).map(cT => ({
          fieldName: cT.id,
          type: (0, _data_grid.getFieldType)(cT.schema)
        })), fieldHistogramsQuery, _field_histograms.DEFAULT_SAMPLER_SHARD_SIZE, combinedRuntimeMappings);
        dataGrid.setColumnCharts(columnChartsData);
      } catch (e) {
        (0, _data_grid.showDataGridColumnChartErrorMessageToast)(e, toastNotifications);
      }
    }
    if (dataGrid.chartsVisible && query !== undefined) {
      fetchColumnChartsData(query);
    }
    // custom comparison
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [dataGrid.chartsVisible, indexPattern.title,
  // eslint-disable-next-line react-hooks/exhaustive-deps
  JSON.stringify([query, dataGrid.visibleColumns, runtimeMappings])]);
  const renderCellValue = (0, _data_grid.useRenderCellValue)(indexPattern, pagination, tableItems);
  return {
    ...dataGrid,
    indexPatternFields,
    renderCellValue,
    timeRangeMs
  };
};
exports.useIndexData = useIndexData;