"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MlEntitySelector = void 0;
var _react = _interopRequireWildcard(require("react"));
var _eui = require("@elastic/eui");
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _useMount = _interopRequireDefault(require("react-use/lib/useMount"));
var _kibana = require("../../contexts/kibana");
var _toast_notification_service = require("../../services/toast_notification_service");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const defaultEntities = {
  anomaly_detector: true,
  data_frame_analytics: true,
  trained_models: true
};

/**
 * Reusable component for picking ML entities.
 *
 * @param entityTypes
 * @param multiSelect
 * @param selectedOptions
 * @param onSelectionChange
 * @param handleDuplicates
 * @constructor
 */
const MlEntitySelector = ({
  entityTypes = defaultEntities,
  multiSelect = true,
  selectedOptions,
  onSelectionChange,
  handleDuplicates = false
}) => {
  const {
    jobs: jobsApi,
    trainedModels,
    dataFrameAnalytics
  } = (0, _kibana.useMlApiContext)();
  const {
    displayErrorToast
  } = (0, _toast_notification_service.useToastNotificationService)();
  const visColorsBehindText = (0, _eui.euiPaletteColorBlindBehindText)();
  const [isLoading, setIsLoading] = (0, _react.useState)(true);
  const [options, setOptions] = (0, _react.useState)([]);
  const fetchOptions = (0, _react.useCallback)(async () => {
    try {
      const newOptions = [];
      if (entityTypes !== null && entityTypes !== void 0 && entityTypes.anomaly_detector) {
        const {
          jobIds: jobIdOptions
        } = await jobsApi.getAllJobAndGroupIds();
        newOptions.push({
          label: _i18n.i18n.translate('xpack.ml.mlEntitySelector.adOptionsLabel', {
            defaultMessage: 'Anomaly detection jobs'
          }),
          isGroupLabelOption: true,
          color: visColorsBehindText[0],
          options: jobIdOptions.map(adId => ({
            label: adId,
            value: adId,
            id: `anomaly_detector:${adId}`,
            key: `anomaly_detector:${adId}`,
            color: visColorsBehindText[0],
            'data-test-subj': `mlAdJobOption`
          }))
        });
      }
      if (entityTypes !== null && entityTypes !== void 0 && entityTypes.data_frame_analytics) {
        const dfa = await dataFrameAnalytics.getDataFrameAnalytics();
        if (dfa.count > 0) {
          newOptions.push({
            label: _i18n.i18n.translate('xpack.ml.mlEntitySelector.dfaOptionsLabel', {
              defaultMessage: 'Data frame analytics'
            }),
            isGroupLabelOption: true,
            options: dfa.data_frame_analytics.map(({
              id: dfaId
            }) => ({
              label: dfaId,
              value: dfaId,
              id: `data_frame_analytics:${dfaId}`,
              key: `data_frame_analytics:${dfaId}`,
              color: visColorsBehindText[2],
              'data-test-subj': `mlDfaJobOption`
            }))
          });
        }
      }
      if (entityTypes !== null && entityTypes !== void 0 && entityTypes.trained_models) {
        const models = await trainedModels.getTrainedModels();
        if (models.length > 0) {
          newOptions.push({
            label: _i18n.i18n.translate('xpack.ml.mlEntitySelector.trainedModelsLabel', {
              defaultMessage: 'Trained models'
            }),
            isGroupLabelOption: true,
            options: models.map(({
              model_id: modelId
            }) => ({
              label: modelId,
              value: modelId,
              id: `trained_models:${modelId}`,
              key: `trained_models:${modelId}`,
              color: visColorsBehindText[3],
              'data-test-subj': `mlTrainedModelOption`
            }))
          });
        }
      }
      setOptions(newOptions);
    } catch (error) {
      displayErrorToast(error, _i18n.i18n.translate('xpack.ml.mlEntitySelector.fetchError', {
        defaultMessage: 'Failed to fetch ML entities'
      }));
    }
    setIsLoading(false);
  }, [jobsApi, trainedModels, dataFrameAnalytics, entityTypes, visColorsBehindText, displayErrorToast]);
  (0, _useMount.default)(function fetchOptionsOnMount() {
    fetchOptions();
  });
  const selectedEntities = (0, _react.useMemo)(() => {
    return (selectedOptions !== null && selectedOptions !== void 0 ? selectedOptions : []).flatMap(o => {
      const fromOptions = options.flatMap(g => g.options).filter(op => op.value === o.id);
      return fromOptions.length > 0 ? fromOptions : [{
        value: o.id,
        label: o.id,
        key: `unknown:${o.id}`,
        'data-test-subj': `mlUnknownOption ${o.id}`
      }];
    });
  }, [options, selectedOptions]);
  const onChange = (0, _react.useCallback)(selection => {
    if (!onSelectionChange) return;
    let resultSelection = selection;
    if (handleDuplicates) {
      const prevCounts = (0, _lodash.countBy)(selectedEntities, 'value');
      const currentCounts = (0, _lodash.countBy)(selection, 'value');
      resultSelection = resultSelection.filter(({
        value
      }) => {
        // If an ID with duplicates has been removed, delete all of them.
        return !(prevCounts[value] > 1 && currentCounts[value] < prevCounts[value]);
      });
    }
    onSelectionChange(resultSelection.map(s => {
      const [type] = s.key.split(':');
      return {
        id: s.value,
        type: type
      };
    }));
  }, [onSelectionChange, selectedEntities, handleDuplicates]);
  return /*#__PURE__*/_react.default.createElement(_eui.EuiComboBox, {
    autoFocus: true,
    isLoading: isLoading,
    singleSelection: !multiSelect,
    selectedOptions: selectedEntities,
    options: options,
    onChange: onChange,
    fullWidth: true,
    "data-test-subj": `mlEntitySelector_${isLoading ? 'loading' : 'loaded'}`,
    isInvalid: false
  });
};
exports.MlEntitySelector = MlEntitySelector;