"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.JobImportService = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function isImportedAdJobs(obj) {
  return Array.isArray(obj) && obj.some(o => o.job && o.datafeed);
}
function isDataFrameAnalyticsConfigs(obj) {
  return Array.isArray(obj) && obj.some(o => o.dest && o.analysis);
}
class JobImportService {
  _readFile(file) {
    return new Promise((resolve, reject) => {
      if (file && file.size) {
        const reader = new FileReader();
        reader.readAsText(file);
        reader.onload = (() => {
          return () => {
            const data = reader.result;
            if (typeof data === 'string') {
              try {
                const json = JSON.parse(data);
                resolve(json);
              } catch (error) {
                reject();
              }
            } else {
              reject();
            }
          };
        })();
      } else {
        reject();
      }
    });
  }
  async readJobConfigs(file) {
    try {
      const json = await this._readFile(file);
      const jobs = Array.isArray(json) ? json : [json];
      if (isImportedAdJobs(jobs)) {
        const jobIds = jobs.map(j => j.job.job_id);
        return {
          jobs,
          jobIds,
          jobType: 'anomaly-detector'
        };
      } else if (isDataFrameAnalyticsConfigs(jobs)) {
        const jobIds = jobs.map(j => j.id);
        return {
          jobs,
          jobIds,
          jobType: 'data-frame-analytics'
        };
      } else {
        return {
          jobs: [],
          jobIds: [],
          jobType: null
        };
      }
    } catch (error) {
      return {
        jobs: [],
        jobIds: [],
        jobType: null
      };
    }
  }
  renameAdJobs(jobIds, jobs) {
    if (jobs.length !== jobs.length) {
      return jobs;
    }
    return jobs.map((j, i) => {
      const {
        jobId
      } = jobIds[i];
      j.job.job_id = jobId;
      j.datafeed.job_id = jobId;
      j.datafeed.datafeed_id = `datafeed-${jobId}`;
      return j;
    });
  }
  renameDfaJobs(jobIds, jobs) {
    if (jobs.length !== jobs.length) {
      return jobs;
    }
    return jobs.map((j, i) => {
      const {
        jobId,
        destIndex
      } = jobIds[i];
      j.id = jobId;
      if (destIndex !== undefined) {
        j.dest.index = destIndex;
      }
      return j;
    });
  }
  async validateJobs(jobs, type, getDataViewTitles, getFilters) {
    const existingDataViews = new Set(await getDataViewTitles());
    const existingFilters = new Set((await getFilters()).map(f => f.filter_id));
    const tempJobs = [];
    const skippedJobs = [];
    const commonJobs = type === 'anomaly-detector' ? jobs.map(j => ({
      jobId: j.job.job_id,
      indices: j.datafeed.indices,
      filters: getFilterIdsFromJob(j.job)
    })) : jobs.map(j => ({
      jobId: j.id,
      destIndex: j.dest.index,
      indices: Array.isArray(j.source.index) ? j.source.index : [j.source.index]
    }));
    commonJobs.forEach(({
      jobId,
      indices,
      filters = [],
      destIndex
    }) => {
      const missingIndices = indices.filter(i => existingDataViews.has(i) === false);
      const missingFilters = filters.filter(i => existingFilters.has(i) === false);
      if (missingIndices.length === 0 && missingFilters.length === 0) {
        tempJobs.push({
          jobId,
          ...(type === 'data-frame-analytics' ? {
            destIndex
          } : {})
        });
      } else {
        skippedJobs.push({
          jobId,
          missingIndices,
          missingFilters
        });
      }
    });
    return {
      jobs: tempJobs,
      skippedJobs
    };
  }
}
exports.JobImportService = JobImportService;
function getFilterIdsFromJob(job) {
  const filters = new Set();
  job.analysis_config.detectors.forEach(d => {
    if (d.custom_rules === undefined) {
      return;
    }
    d.custom_rules.forEach(r => {
      if (r.scope === undefined) {
        return;
      }
      Object.values(r.scope).forEach(s => {
        filters.add(s.filter_id);
      });
    });
  });
  return [...filters];
}