"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CustomUrlList = void 0;
var _react = _interopRequireWildcard(require("react"));
var _eui = require("@elastic/eui");
var _i18n = require("@kbn/i18n");
var _i18nReact = require("@kbn/i18n-react");
var _kibana = require("../../../contexts/kibana");
var _custom_url_utils = require("../../../util/custom_url_utils");
var _utils = require("./utils");
var _parse_interval = require("../../../../../common/util/parse_interval");
var _constants = require("./constants");
var _anomaly_detection_jobs = require("../../../../../common/types/anomaly_detection_jobs");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function isValidTimeRange(timeRange) {
  // Allow empty timeRange string, which gives the 'auto' behaviour.
  if (timeRange === undefined || timeRange.length === 0 || timeRange === _constants.TIME_RANGE_TYPE.AUTO) {
    return true;
  }
  const interval = (0, _parse_interval.parseInterval)(timeRange);
  return interval !== null;
}
/*
 * React component for listing the custom URLs added to a job,
 * with buttons for testing and deleting each custom URL.
 */
const CustomUrlList = ({
  job,
  customUrls,
  onChange: setCustomUrls
}) => {
  const {
    services: {
      http,
      notifications
    }
  } = (0, _kibana.useMlKibana)();
  const [expandedUrlIndex, setExpandedUrlIndex] = (0, _react.useState)(null);
  const onLabelChange = (e, index) => {
    if (index < customUrls.length) {
      customUrls[index] = {
        ...customUrls[index],
        url_name: e.target.value
      };
      setCustomUrls([...customUrls]);
    }
  };
  const onUrlValueChange = (e, index) => {
    if (index < customUrls.length) {
      customUrls[index] = {
        ...customUrls[index],
        url_value: e.target.value
      };
      setCustomUrls([...customUrls]);
    }
  };
  const onTimeRangeChange = (e, index) => {
    if (index < customUrls.length) {
      customUrls[index] = {
        ...customUrls[index]
      };
      const timeRange = e.target.value;
      if (timeRange !== undefined && timeRange.length > 0) {
        customUrls[index].time_range = timeRange;
      } else {
        delete customUrls[index].time_range;
      }
      setCustomUrls([...customUrls]);
    }
  };
  const onDeleteButtonClick = index => {
    if (index < customUrls.length) {
      customUrls.splice(index, 1);
      setCustomUrls([...customUrls]);
    }
  };
  const onTestButtonClick = index => {
    if (index < customUrls.length) {
      (0, _utils.getTestUrl)(job, customUrls[index]).then(testUrl => {
        (0, _custom_url_utils.openCustomUrlWindow)(testUrl, customUrls[index], http.basePath.get());
      }).catch(resp => {
        // eslint-disable-next-line no-console
        console.error('Error obtaining URL for test:', resp);
        const {
          toasts
        } = notifications;
        toasts.addDanger(_i18n.i18n.translate('xpack.ml.customUrlEditorList.obtainingUrlToTestConfigurationErrorMessage', {
          defaultMessage: 'An error occurred obtaining the URL to test the configuration'
        }));
      });
    }
  };
  const customUrlRows = customUrls.map((customUrl, index) => {
    // Validate the label.
    const label = customUrl.url_name;
    const otherUrls = [...customUrls];
    otherUrls.splice(index, 1); // Don't compare label with itself.
    const isInvalidLabel = !(0, _custom_url_utils.isValidLabel)(label, otherUrls);
    const invalidLabelError = isInvalidLabel ? [_i18n.i18n.translate('xpack.ml.customUrlEditorList.labelIsNotUniqueErrorMessage', {
      defaultMessage: 'A unique label must be supplied'
    })] : [];

    // Validate the time range.
    const timeRange = customUrl.time_range;
    const isInvalidTimeRange = !isValidTimeRange(timeRange);
    const invalidIntervalError = isInvalidTimeRange ? [_i18n.i18n.translate('xpack.ml.customUrlEditorList.invalidTimeRangeFormatErrorMessage', {
      defaultMessage: 'Invalid format'
    })] : [];
    return /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexGroup, {
      key: `url_${index}`,
      "data-test-subj": `mlJobEditCustomUrlItem_${index}`
    }, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
      grow: false
    }, /*#__PURE__*/_react.default.createElement(_eui.EuiFormRow, {
      label: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
        id: "xpack.ml.customUrlEditorList.labelLabel",
        defaultMessage: "Label"
      }),
      isInvalid: isInvalidLabel,
      error: invalidLabelError,
      "data-test-subj": "mlJobEditCustomUrlItemLabel"
    }, /*#__PURE__*/_react.default.createElement(_eui.EuiFieldText, {
      value: label,
      isInvalid: isInvalidLabel,
      onChange: e => onLabelChange(e, index),
      "data-test-subj": `mlJobEditCustomUrlLabelInput_${index}`
    }))), /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, null, /*#__PURE__*/_react.default.createElement(_eui.EuiFormRow, {
      fullWidth: true,
      label: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
        id: "xpack.ml.customUrlEditorList.urlLabel",
        defaultMessage: "URL"
      })
    }, index === expandedUrlIndex ? /*#__PURE__*/_react.default.createElement(_eui.EuiTextArea, {
      inputRef: input => {
        if (input) {
          input.focus();
        }
      },
      fullWidth: true,
      value: customUrl.url_value,
      onChange: e => onUrlValueChange(e, index),
      onBlur: () => {
        setExpandedUrlIndex(null);
      },
      "data-test-subj": `mlJobEditCustomUrlTextarea_${index}`
    }) : /*#__PURE__*/_react.default.createElement(_eui.EuiFieldText, {
      fullWidth: true,
      value: customUrl.url_value,
      readOnly: true,
      onFocus: () => setExpandedUrlIndex(index),
      "data-test-subj": `mlJobEditCustomUrlInput_${index}`
    }))), (0, _anomaly_detection_jobs.isAnomalyDetectionJob)(job) ? /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
      grow: false
    }, /*#__PURE__*/_react.default.createElement(_eui.EuiFormRow, {
      label: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
        id: "xpack.ml.customUrlEditorList.timeRangeLabel",
        defaultMessage: "Time range"
      }),
      error: invalidIntervalError,
      isInvalid: isInvalidTimeRange
    }, /*#__PURE__*/_react.default.createElement(_eui.EuiFieldText, {
      value: customUrl.time_range || '',
      isInvalid: isInvalidTimeRange,
      placeholder: _constants.TIME_RANGE_TYPE.AUTO,
      onChange: e => onTimeRangeChange(e, index)
    }))) : null, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
      grow: false
    }, /*#__PURE__*/_react.default.createElement(_eui.EuiFormRow, {
      hasEmptyLabelSpace: true
    }, /*#__PURE__*/_react.default.createElement(_eui.EuiToolTip, {
      content: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
        id: "xpack.ml.customUrlEditorList.testCustomUrlTooltip",
        defaultMessage: "Test custom URL"
      })
    }, /*#__PURE__*/_react.default.createElement(_eui.EuiButtonIcon, {
      size: "s",
      color: "primary",
      onClick: () => onTestButtonClick(index),
      iconType: "popout",
      "aria-label": _i18n.i18n.translate('xpack.ml.customUrlEditorList.testCustomUrlAriaLabel', {
        defaultMessage: 'Test custom URL'
      }),
      "data-test-subj": "mlJobEditTestCustomUrlButton"
    })))), /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
      grow: false
    }, /*#__PURE__*/_react.default.createElement(_eui.EuiFormRow, {
      hasEmptyLabelSpace: true
    }, /*#__PURE__*/_react.default.createElement(_eui.EuiToolTip, {
      content: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
        id: "xpack.ml.customUrlEditorList.deleteCustomUrlTooltip",
        defaultMessage: "Delete custom URL"
      })
    }, /*#__PURE__*/_react.default.createElement(_eui.EuiButtonIcon, {
      size: "s",
      color: "danger",
      onClick: () => onDeleteButtonClick(index),
      iconType: "trash",
      "aria-label": _i18n.i18n.translate('xpack.ml.customUrlEditorList.deleteCustomUrlAriaLabel', {
        defaultMessage: 'Delete custom URL'
      }),
      "data-test-subj": `mlJobEditDeleteCustomUrlButton_${index}`
    }))))), /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
      size: "m"
    }));
  });
  return /*#__PURE__*/_react.default.createElement("div", {
    "data-test-subj": "mlJobEditCustomUrlsList"
  }, customUrlRows);
};
exports.CustomUrlList = CustomUrlList;