"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TileStatusTracker = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = _interopRequireDefault(require("lodash"));
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _constants = require("../../../../common/constants");
var _geo_tile_utils = require("../../../classes/util/geo_tile_utils");
var _tile_meta_feature_utils = require("../../../classes/util/tile_meta_feature_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class TileStatusTracker extends _react.Component {
  constructor(...args) {
    super(...args);
    (0, _defineProperty2.default)(this, "_isMounted", false);
    // Tile cache tracks active tile requests
    // 'sourcedataloading' event adds tile request to cache
    // 'sourcedata' and 'error' events remove tile request from cache
    // Tile requests with 'aborted' status are removed from cache when reporting 'areTilesLoaded' status
    (0, _defineProperty2.default)(this, "_tileCache", []);
    // Tile error cache tracks tile request errors per layer
    // Error cache is cleared when map center tile changes
    (0, _defineProperty2.default)(this, "_tileErrorCache", {});
    // Layer cache tracks layers that have requested one or more tiles
    // Layer cache is used so that only a layer that has requested one or more tiles reports 'areTilesLoaded' status
    // layer cache is never cleared
    (0, _defineProperty2.default)(this, "_layerCache", new Map());
    (0, _defineProperty2.default)(this, "_prevCenterTileKey", void 0);
    (0, _defineProperty2.default)(this, "_onSourceDataLoading", e => {
      if (e.sourceId && e.sourceId !== _constants.SPATIAL_FILTERS_LAYER_ID && e.dataType === 'source' && e.tile && (e.source.type === 'vector' || e.source.type === 'raster')) {
        const targetLayer = this.props.layerList.find(layer => {
          return layer.ownsMbSourceId(e.sourceId);
        });
        const layerId = targetLayer ? targetLayer.getId() : undefined;
        if (layerId && !this._layerCache.has(layerId)) {
          this._layerCache.set(layerId, true);
        }
        const tracked = this._tileCache.find(tile => {
          return tile.mbKey === e.tile.tileID.key && tile.mbSourceId === e.sourceId;
        });
        if (!tracked) {
          this._tileCache.push({
            mbKey: e.tile.tileID.key,
            mbSourceId: e.sourceId,
            mbTile: e.tile
          });
          this._updateTileStatusForAllLayers();
        }
      }
    });
    (0, _defineProperty2.default)(this, "_onError", e => {
      if (e.sourceId && e.sourceId !== _constants.SPATIAL_FILTERS_LAYER_ID && e.tile && (e.source.type === 'vector' || e.source.type === 'raster')) {
        const targetLayer = this.props.layerList.find(layer => {
          return layer.ownsMbSourceId(e.sourceId);
        });
        const layerId = targetLayer ? targetLayer.getId() : undefined;
        if (layerId) {
          const layerErrors = this._tileErrorCache[layerId] ? this._tileErrorCache[layerId] : [];
          layerErrors.push({
            ...e.error,
            tileZXYKey: `${e.tile.tileID.canonical.z}/${e.tile.tileID.canonical.x}/${e.tile.tileID.canonical.y}`
          });
          this._tileErrorCache[layerId] = layerErrors;
        }
        this._removeTileFromCache(e.sourceId, e.tile.tileID.key);
      }
    });
    (0, _defineProperty2.default)(this, "_onSourceData", e => {
      if (e.sourceId && e.sourceId !== _constants.SPATIAL_FILTERS_LAYER_ID && e.dataType === 'source' && e.tile && (e.source.type === 'vector' || e.source.type === 'raster')) {
        this._removeTileFromCache(e.sourceId, e.tile.tileID.key);
      }
    });
    /*
     * Clear errors when center tile changes.
     * Tracking center tile provides the cleanest way to know when a new data fetching cycle is beginning
     */
    (0, _defineProperty2.default)(this, "_onMove", () => {
      const center = this.props.mbMap.getCenter();
      // Maplibre rounds zoom when 'source.roundZoom' is true and floors zoom when 'source.roundZoom' is false
      // 'source.roundZoom' is true for raster and video layers
      // 'source.roundZoom' is false for vector layers
      // Always floor zoom to keep logic as simple as possible and not have to track center tile by source.
      // We are mainly concerned with showing errors from Elasticsearch vector tile requests (which are vector sources)
      const centerTileKey = (0, _geo_tile_utils.getTileKey)(center.lat, center.lng, Math.floor(this.props.mbMap.getZoom()));
      if (this._prevCenterTileKey !== centerTileKey) {
        this._prevCenterTileKey = centerTileKey;
        this._tileErrorCache = {};
      }
    });
    (0, _defineProperty2.default)(this, "_updateTileStatusForAllLayers", _lodash.default.debounce(() => {
      if (!this._isMounted) {
        return;
      }
      this._tileCache = this._tileCache.filter(tile => {
        return typeof tile.mbTile.aborted === 'boolean' ? !tile.mbTile.aborted : true;
      });
      const layerList = this.props.layerList;
      for (let i = 0; i < layerList.length; i++) {
        const layer = layerList[i];
        if (!this._layerCache.has(layer.getId())) {
          // do not report status for layers that have not started loading tiles.
          continue;
        }
        let atLeastOnePendingTile = false;
        for (let j = 0; j < this._tileCache.length; j++) {
          const tile = this._tileCache[j];
          if (layer.ownsMbSourceId(tile.mbSourceId)) {
            atLeastOnePendingTile = true;
            break;
          }
        }
        const tileErrorMessages = this._tileErrorCache[layer.getId()] ? this._tileErrorCache[layer.getId()].map(tileError => {
          return _i18n.i18n.translate('xpack.maps.tileStatusTracker.tileErrorMsg', {
            defaultMessage: `tile '{tileZXYKey}' failed to load: '{status} {message}'`,
            values: {
              tileZXYKey: tileError.tileZXYKey,
              status: tileError.status,
              message: tileError.message
            }
          });
        }) : [];
        this._updateTileStatusForLayer(layer, !atLeastOnePendingTile, tileErrorMessages.length ? _i18n.i18n.translate('xpack.maps.tileStatusTracker.layerErrorMsg', {
          defaultMessage: `Unable to load {count} tiles: {tileErrors}`,
          values: {
            count: tileErrorMessages.length,
            tileErrors: tileErrorMessages.join(', ')
          }
        }) : undefined);
      }
    }, 100));
    (0, _defineProperty2.default)(this, "_updateTileStatusForLayer", (layer, areTilesLoaded, errorMessage) => {
      this.props.setAreTilesLoaded(layer.getId(), areTilesLoaded);
      if (errorMessage) {
        this.props.setTileLoadError(layer.getId(), errorMessage);
      } else {
        this.props.clearTileLoadError(layer.getId());
      }
      const source = layer.getSource();
      if (layer.isVisible() && source.isESSource() && typeof source.isMvt === 'function' && source.isMvt()) {
        // querySourceFeatures can return duplicated features when features cross tile boundaries.
        // Tile meta will never have duplicated features since by their nature, tile meta is a feature contained within a single tile
        const mbFeatures = this.props.mbMap.querySourceFeatures(layer.getMbSourceId(), {
          sourceLayer: _tile_meta_feature_utils.ES_MVT_META_LAYER_NAME,
          filter: []
        });
        const features = mbFeatures.map(mbFeature => {
          try {
            return {
              type: 'Feature',
              id: mbFeature === null || mbFeature === void 0 ? void 0 : mbFeature.id,
              geometry: mbFeature === null || mbFeature === void 0 ? void 0 : mbFeature.geometry,
              // this getter might throw with non-conforming geometries
              properties: mbFeature === null || mbFeature === void 0 ? void 0 : mbFeature.properties
            };
          } catch (e) {
            return null;
          }
        }).filter(mbFeature => mbFeature !== null);
        this.props.updateMetaFromTiles(layer.getId(), features);
      }
    });
    (0, _defineProperty2.default)(this, "_removeTileFromCache", (mbSourceId, mbKey) => {
      const trackedIndex = this._tileCache.findIndex(tile => {
        return tile.mbKey === mbKey && tile.mbSourceId === mbSourceId;
      });
      if (trackedIndex >= 0) {
        this._tileCache.splice(trackedIndex, 1);
        this._updateTileStatusForAllLayers();
      }
    });
  }
  componentDidMount() {
    this._isMounted = true;
    this.props.mbMap.on('sourcedataloading', this._onSourceDataLoading);
    this.props.mbMap.on('error', this._onError);
    this.props.mbMap.on('sourcedata', this._onSourceData);
    this.props.mbMap.on('move', this._onMove);
  }
  componentWillUnmount() {
    this._isMounted = false;
    this.props.mbMap.off('error', this._onError);
    this.props.mbMap.off('sourcedata', this._onSourceData);
    this.props.mbMap.off('sourcedataloading', this._onSourceDataLoading);
    this.props.mbMap.off('move', this._onMove);
    this._tileCache.length = 0;
  }
  render() {
    return null;
  }
}
exports.TileStatusTracker = TileStatusTracker;