"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.geoLineTitle = exports.REQUIRES_GOLD_LICENSE_MSG = exports.ESGeoLineSource = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = _interopRequireDefault(require("lodash"));
var _react = _interopRequireDefault(require("react"));
var _i18n = require("@kbn/i18n");
var _esQuery = require("@kbn/es-query");
var _constants = require("../../../../common/constants");
var _elasticsearch_util = require("../../../../common/elasticsearch_util");
var _i18n_getters = require("../../../../common/i18n_getters");
var _es_agg_source = require("../es_agg_source");
var _convert_to_geojson = require("./convert_to_geojson");
var _es_doc_field = require("../../fields/es_doc_field");
var _update_source_editor = require("./update_source_editor");
var _valid_string_config = require("../../util/valid_string_config");
var _tooltip_property = require("../../tooltips/tooltip_property");
var _licensed_features = require("../../../licensed_features");
var _execution_context_utils = require("../execution_context_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_TRACKS = 250;
const geoLineTitle = _i18n.i18n.translate('xpack.maps.source.esGeoLineTitle', {
  defaultMessage: 'Tracks'
});
exports.geoLineTitle = geoLineTitle;
const REQUIRES_GOLD_LICENSE_MSG = _i18n.i18n.translate('xpack.maps.source.esGeoLineDisabledReason', {
  defaultMessage: '{title} requires a Gold license.',
  values: {
    title: geoLineTitle
  }
});
exports.REQUIRES_GOLD_LICENSE_MSG = REQUIRES_GOLD_LICENSE_MSG;
class ESGeoLineSource extends _es_agg_source.AbstractESAggSource {
  static createDescriptor(descriptor) {
    const normalizedDescriptor = _es_agg_source.AbstractESAggSource.createDescriptor(descriptor);
    if (!(0, _valid_string_config.isValidStringConfig)(normalizedDescriptor.geoField)) {
      throw new Error('Cannot create an ESGeoLineSource without a geoField');
    }
    if (!(0, _valid_string_config.isValidStringConfig)(normalizedDescriptor.splitField)) {
      throw new Error('Cannot create an ESGeoLineSource without a splitField');
    }
    if (!(0, _valid_string_config.isValidStringConfig)(normalizedDescriptor.sortField)) {
      throw new Error('Cannot create an ESGeoLineSource without a sortField');
    }
    return {
      ...normalizedDescriptor,
      type: _constants.SOURCE_TYPES.ES_GEO_LINE,
      geoField: normalizedDescriptor.geoField,
      splitField: normalizedDescriptor.splitField,
      sortField: normalizedDescriptor.sortField
    };
  }
  constructor(descriptor) {
    const sourceDescriptor = ESGeoLineSource.createDescriptor(descriptor);
    super(sourceDescriptor);
    (0, _defineProperty2.default)(this, "_descriptor", void 0);
    this._descriptor = sourceDescriptor;
  }
  getBucketsName() {
    return _i18n.i18n.translate('xpack.maps.source.esGeoLine.bucketsName', {
      defaultMessage: 'tracks'
    });
  }
  renderSourceSettingsEditor({
    onChange
  }) {
    return /*#__PURE__*/_react.default.createElement(_update_source_editor.UpdateSourceEditor, {
      bucketsName: this.getBucketsName(),
      indexPatternId: this.getIndexPatternId(),
      onChange: onChange,
      metrics: this._descriptor.metrics,
      splitField: this._descriptor.splitField,
      sortField: this._descriptor.sortField
    });
  }
  getSyncMeta() {
    return {
      splitField: this._descriptor.splitField,
      sortField: this._descriptor.sortField
    };
  }
  async getImmutableProperties() {
    return [{
      label: (0, _i18n_getters.getDataSourceLabel)(),
      value: geoLineTitle
    }, {
      label: (0, _i18n_getters.getDataViewLabel)(),
      value: await this.getDisplayName()
    }, {
      label: _i18n.i18n.translate('xpack.maps.source.esGeoLine.geospatialFieldLabel', {
        defaultMessage: 'Geospatial field'
      }),
      value: this._descriptor.geoField
    }];
  }
  _createSplitField() {
    return new _es_doc_field.ESDocField({
      fieldName: this._descriptor.splitField,
      source: this,
      origin: _constants.FIELD_ORIGIN.SOURCE
    });
  }
  getFieldNames() {
    return [...this.getMetricFields().map(esAggMetricField => esAggMetricField.getName()), this._descriptor.splitField, this._descriptor.sortField];
  }
  async getFields() {
    return [...this.getMetricFields(), this._createSplitField()];
  }
  getFieldByName(name) {
    return name === this._descriptor.splitField ? this._createSplitField() : this.getMetricFieldForName(name);
  }
  isGeoGridPrecisionAware() {
    return false;
  }
  showJoinEditor() {
    return false;
  }
  async getGeoJsonWithMeta(layerName, requestMeta, registerCancelCallback, isRequestStillActive, inspectorAdapters) {
    if (!(0, _licensed_features.getIsGoldPlus)()) {
      throw new Error(REQUIRES_GOLD_LICENSE_MSG);
    }
    const indexPattern = await this.getIndexPattern();

    // Request is broken into 2 requests
    // 1) fetch entities: filtered by buffer so that top entities in view are returned
    // 2) fetch tracks: not filtered by buffer to avoid having invalid tracks
    //    when the track extends beyond the area of the map buffer.

    //
    // Fetch entities
    //
    const entitySearchSource = await this.makeSearchSource(requestMeta, 0);
    entitySearchSource.setField('trackTotalHits', false);
    const splitField = (0, _elasticsearch_util.getField)(indexPattern, this._descriptor.splitField);
    const cardinalityAgg = {
      precision_threshold: 1
    };
    const termsAgg = {
      size: MAX_TRACKS
    };
    entitySearchSource.setField('aggs', {
      totalEntities: {
        cardinality: (0, _elasticsearch_util.addFieldToDSL)(cardinalityAgg, splitField)
      },
      entitySplit: {
        terms: (0, _elasticsearch_util.addFieldToDSL)(termsAgg, splitField)
      }
    });
    if (splitField.type === 'string') {
      const entityIsNotEmptyFilter = (0, _esQuery.buildPhraseFilter)(splitField, '', indexPattern);
      entityIsNotEmptyFilter.meta.negate = true;
      entitySearchSource.setField('filter', [...entitySearchSource.getField('filter'), entityIsNotEmptyFilter]);
    }
    const entityResp = await this._runEsQuery({
      requestId: `${this.getId()}_entities`,
      requestName: _i18n.i18n.translate('xpack.maps.source.esGeoLine.entityRequestName', {
        defaultMessage: '{layerName} entities',
        values: {
          layerName
        }
      }),
      searchSource: entitySearchSource,
      registerCancelCallback,
      requestDescription: _i18n.i18n.translate('xpack.maps.source.esGeoLine.entityRequestDescription', {
        defaultMessage: 'Elasticsearch terms request to fetch entities within map buffer.'
      }),
      searchSessionId: requestMeta.searchSessionId,
      executionContext: (0, _execution_context_utils.mergeExecutionContext)({
        description: 'es_geo_line:entities'
      }, requestMeta.executionContext),
      requestsAdapter: inspectorAdapters.requests
    });
    const entityBuckets = _lodash.default.get(entityResp, 'aggregations.entitySplit.buckets', []);
    const totalEntities = _lodash.default.get(entityResp, 'aggregations.totalEntities.value', 0);
    const areEntitiesTrimmed = entityBuckets.length >= MAX_TRACKS;
    if (totalEntities === 0) {
      return {
        data: _constants.EMPTY_FEATURE_COLLECTION,
        meta: {
          areResultsTrimmed: false,
          areEntitiesTrimmed: false,
          entityCount: 0,
          numTrimmedTracks: 0,
          totalEntities: 0
        }
      };
    }

    //
    // Fetch tracks
    //
    const entityFilters = {};
    for (let i = 0; i < entityBuckets.length; i++) {
      entityFilters[entityBuckets[i].key] = (0, _esQuery.buildPhraseFilter)(splitField, entityBuckets[i].key, indexPattern).query;
    }
    const tracksSearchFilters = {
      ...requestMeta
    };
    delete tracksSearchFilters.buffer;
    const tracksSearchSource = await this.makeSearchSource(tracksSearchFilters, 0);
    tracksSearchSource.setField('trackTotalHits', false);
    tracksSearchSource.setField('aggs', {
      tracks: {
        filters: {
          filters: entityFilters
        },
        aggs: {
          path: {
            geo_line: {
              point: {
                field: this._descriptor.geoField
              },
              sort: {
                field: this._descriptor.sortField
              }
            }
          },
          ...this.getValueAggsDsl(indexPattern)
        }
      }
    });
    const tracksResp = await this._runEsQuery({
      requestId: `${this.getId()}_tracks`,
      requestName: _i18n.i18n.translate('xpack.maps.source.esGeoLine.trackRequestName', {
        defaultMessage: '{layerName} tracks',
        values: {
          layerName
        }
      }),
      searchSource: tracksSearchSource,
      registerCancelCallback,
      requestDescription: _i18n.i18n.translate('xpack.maps.source.esGeoLine.trackRequestDescription', {
        defaultMessage: 'Elasticsearch geo_line request to fetch tracks for entities. Tracks are not filtered by map buffer.'
      }),
      searchSessionId: requestMeta.searchSessionId,
      executionContext: (0, _execution_context_utils.mergeExecutionContext)({
        description: 'es_geo_line:tracks'
      }, requestMeta.executionContext),
      requestsAdapter: inspectorAdapters.requests
    });
    const {
      featureCollection,
      numTrimmedTracks
    } = (0, _convert_to_geojson.convertToGeoJson)(tracksResp, this._descriptor.splitField);
    return {
      data: featureCollection,
      meta: {
        // meta.areResultsTrimmed is used by updateDueToExtent to skip re-fetching results
        // when extent changes contained by original extent are not needed
        // Only trigger re-fetch when the number of entities are trimmed
        // Do not trigger re-fetch when tracks are trimmed since the tracks themselves are not filtered by map view extent.
        areResultsTrimmed: areEntitiesTrimmed,
        areEntitiesTrimmed,
        entityCount: entityBuckets.length,
        numTrimmedTracks,
        totalEntities
      }
    };
  }
  getSourceStatus(sourceDataRequest) {
    const featureCollection = sourceDataRequest ? sourceDataRequest.getData() : null;
    const meta = sourceDataRequest ? sourceDataRequest.getMeta() : null;
    if (!featureCollection || !meta) {
      // no tooltip content needed when there is no feature collection or meta
      return {
        tooltipContent: null,
        areResultsTrimmed: false
      };
    }
    const entitiesFoundMsg = meta.areEntitiesTrimmed ? _i18n.i18n.translate('xpack.maps.esGeoLine.areEntitiesTrimmedMsg', {
      defaultMessage: `Results limited to first {entityCount} tracks of ~{totalEntities}.`,
      values: {
        entityCount: meta.entityCount.toLocaleString(),
        totalEntities: meta.totalEntities.toLocaleString()
      }
    }) : _i18n.i18n.translate('xpack.maps.esGeoLine.tracksCountMsg', {
      defaultMessage: `Found {entityCount} tracks.`,
      values: {
        entityCount: meta.entityCount.toLocaleString()
      }
    });
    const tracksTrimmedMsg = meta.numTrimmedTracks > 0 ? _i18n.i18n.translate('xpack.maps.esGeoLine.tracksTrimmedMsg', {
      defaultMessage: `{numTrimmedTracks} of {entityCount} tracks are incomplete.`,
      values: {
        entityCount: meta.entityCount.toLocaleString(),
        numTrimmedTracks: meta.numTrimmedTracks.toLocaleString()
      }
    }) : undefined;
    return {
      tooltipContent: tracksTrimmedMsg ? `${entitiesFoundMsg} ${tracksTrimmedMsg}` : entitiesFoundMsg,
      // Used to show trimmed icon in legend. Trimmed icon signals the following
      // 1) number of entities are trimmed.
      // 2) one or more tracks are incomplete.
      areResultsTrimmed: meta.areEntitiesTrimmed || meta.numTrimmedTracks > 0
    };
  }
  isFilterByMapBounds() {
    return true;
  }
  hasTooltipProperties() {
    return true;
  }
  async getSupportedShapeTypes() {
    return [_constants.VECTOR_SHAPE_TYPE.LINE];
  }
  async getTooltipProperties(properties) {
    const tooltipProperties = await super.getTooltipProperties(properties);
    tooltipProperties.push(new _tooltip_property.TooltipProperty('isTrackComplete', _i18n.i18n.translate('xpack.maps.source.esGeoLine.isTrackCompleteLabel', {
      defaultMessage: 'track is complete'
    }), properties.complete.toString()));
    return tooltipProperties;
  }
  async getLicensedFeatures() {
    return [_licensed_features.LICENSED_FEATURES.GEO_LINE_AGG];
  }
}
exports.ESGeoLineSource = ESGeoLineSource;