"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.NO_RESULTS_ICON_AND_TOOLTIPCONTENT = exports.AbstractVectorLayer = void 0;
exports.isVectorLayer = isVectorLayer;
exports.noResultsIcon = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _react = _interopRequireDefault(require("react"));
var _uuid = require("uuid");
var _lodash = _interopRequireDefault(require("lodash"));
var _eui = require("@elastic/eui");
var _i18n = require("@kbn/i18n");
var _layer = require("../layer");
var _vector_style = require("../../styles/vector/vector_style");
var _constants = require("../../../../common/constants");
var _join_tooltip_property = require("../../tooltips/join_tooltip_property");
var _data_request = require("../../util/data_request");
var _can_skip_fetch = require("../../util/can_skip_fetch");
var _mb_filter_expressions = require("../../util/mb_filter_expressions");
var _build_vector_request_meta = require("../build_vector_request_meta");
var _get_agg_key = require("../../../../common/get_agg_key");
var _bounds_data = require("./bounds_data");
var _mask = require("./mask");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SUPPORTS_FEATURE_EDITING_REQUEST_ID = 'SUPPORTS_FEATURE_EDITING_REQUEST_ID';
function isVectorLayer(layer) {
  return layer.canShowTooltip !== undefined;
}
const noResultsIcon = /*#__PURE__*/_react.default.createElement(_eui.EuiIcon, {
  size: "m",
  color: "subdued",
  type: "minusInCircle"
});
exports.noResultsIcon = noResultsIcon;
const NO_RESULTS_ICON_AND_TOOLTIPCONTENT = {
  icon: noResultsIcon,
  tooltipContent: _i18n.i18n.translate('xpack.maps.vectorLayer.noResultsFoundTooltip', {
    defaultMessage: `No results found.`
  })
};
exports.NO_RESULTS_ICON_AND_TOOLTIPCONTENT = NO_RESULTS_ICON_AND_TOOLTIPCONTENT;
class AbstractVectorLayer extends _layer.AbstractLayer {
  static createDescriptor(options, mapColors) {
    var _options$disableToolt;
    const layerDescriptor = super.createDescriptor(options);
    layerDescriptor.type = layerDescriptor.type !== undefined ? layerDescriptor.type : _constants.LAYER_TYPE.GEOJSON_VECTOR;
    if (!options.style) {
      const styleProperties = _vector_style.VectorStyle.createDefaultStyleProperties(mapColors ? mapColors : []);
      layerDescriptor.style = _vector_style.VectorStyle.createDescriptor(styleProperties);
    }
    if (!options.joins) {
      layerDescriptor.joins = [];
    }
    layerDescriptor.disableTooltips = (_options$disableToolt = options.disableTooltips) !== null && _options$disableToolt !== void 0 ? _options$disableToolt : false;
    return layerDescriptor;
  }
  constructor({
    layerDescriptor,
    source,
    joins = [],
    customIcons,
    chartsPaletteServiceGetColor
  }) {
    super({
      layerDescriptor,
      source
    });
    (0, _defineProperty2.default)(this, "_style", void 0);
    (0, _defineProperty2.default)(this, "_joins", void 0);
    (0, _defineProperty2.default)(this, "_descriptor", void 0);
    (0, _defineProperty2.default)(this, "_masks", void 0);
    this._joins = joins;
    this._descriptor = AbstractVectorLayer.createDescriptor(layerDescriptor);
    this._style = new _vector_style.VectorStyle(this._descriptor.style, source, this, customIcons, chartsPaletteServiceGetColor);
    this._masks = this._createMasks();
  }
  async cloneDescriptor() {
    const clones = await super.cloneDescriptor();
    if (clones.length === 0) {
      return [];
    }
    const clonedDescriptor = clones[0];
    if (clonedDescriptor.joins) {
      clonedDescriptor.joins.forEach(joinDescriptor => {
        if (joinDescriptor.right && joinDescriptor.right.type === _constants.SOURCE_TYPES.TABLE_SOURCE) {
          throw new Error('Cannot clone table-source. Should only be used in MapEmbeddable, not in UX');
        }
        const termSourceDescriptor = joinDescriptor.right;

        // todo: must tie this to generic thing
        const originalJoinId = joinDescriptor.right.id;

        // right.id is uuid used to track requests in inspector
        joinDescriptor.right.id = (0, _uuid.v4)();

        // Update all data driven styling properties using join fields
        if (clonedDescriptor.style && 'properties' in clonedDescriptor.style) {
          const metrics = termSourceDescriptor.metrics && termSourceDescriptor.metrics.length ? termSourceDescriptor.metrics : [{
            type: _constants.AGG_TYPE.COUNT
          }];
          metrics.forEach(metricsDescriptor => {
            const originalJoinKey = (0, _get_agg_key.getJoinAggKey)({
              aggType: metricsDescriptor.type,
              aggFieldName: 'field' in metricsDescriptor ? metricsDescriptor.field : '',
              rightSourceId: originalJoinId
            });
            const newJoinKey = (0, _get_agg_key.getJoinAggKey)({
              aggType: metricsDescriptor.type,
              aggFieldName: 'field' in metricsDescriptor ? metricsDescriptor.field : '',
              rightSourceId: joinDescriptor.right.id
            });
            Object.keys(clonedDescriptor.style.properties).forEach(key => {
              const styleProp = clonedDescriptor.style.properties[key];
              if ('type' in styleProp && styleProp.type === _constants.STYLE_TYPE.DYNAMIC) {
                const options = styleProp.options;
                if (options.field && options.field.origin === _constants.FIELD_ORIGIN.JOIN && options.field.name === originalJoinKey) {
                  options.field.name = newJoinKey;
                }
              }
            });
          });
        }
      });
    }
    return [clonedDescriptor];
  }
  getSource() {
    return super.getSource();
  }
  getStyleForEditing() {
    return this._style;
  }
  getStyle() {
    return this._style;
  }
  getCurrentStyle() {
    return this._style;
  }
  getJoins() {
    return this._joins.slice();
  }
  getJoinsDisabledReason() {
    return this.getSource().getJoinsDisabledReason();
  }
  getValidJoins() {
    return this.getJoins().filter(join => {
      return join.hasCompleteConfig();
    });
  }
  supportsFeatureEditing() {
    const dataRequest = this.getDataRequest(SUPPORTS_FEATURE_EDITING_REQUEST_ID);
    const data = dataRequest === null || dataRequest === void 0 ? void 0 : dataRequest.getData();
    return data ? data.supportsFeatureEditing : false;
  }
  hasJoins() {
    return this.getValidJoins().length > 0;
  }
  showJoinEditor() {
    return this.getSource().showJoinEditor();
  }
  isLayerLoading(zoom) {
    const isSourceLoading = super.isLayerLoading(zoom);
    if (isSourceLoading) {
      return true;
    }
    return this.getValidJoins().some(join => {
      const joinDataRequest = this.getDataRequest(join.getSourceDataRequestId());
      return !joinDataRequest || joinDataRequest.isLoading();
    });
  }
  getLayerIcon(isTocIcon) {
    throw new Error('Should implement AbstractVectorLayer#getLayerIcon');
  }
  getLayerTypeIconName() {
    return 'vector';
  }
  async hasLegendDetails() {
    return this.getCurrentStyle().hasLegendDetails();
  }
  renderLegendDetails() {
    return this.getCurrentStyle().renderLegendDetails();
  }
  async getBounds(getDataRequestContext) {
    return (0, _bounds_data.syncBoundsData)({
      layerId: this.getId(),
      syncContext: getDataRequestContext(this.getId()),
      source: this.getSource(),
      sourceQuery: this.getQuery()
    });
  }
  async getLeftJoinFields() {
    return await this.getSource().getLeftJoinFields();
  }
  _getJoinFields() {
    const joinFields = [];
    this.getValidJoins().forEach(join => {
      const fields = join.getJoinFields();
      joinFields.push(...fields);
    });
    return joinFields;
  }
  async getFields() {
    const sourceFields = await this.getSource().getFields();
    return [...sourceFields, ...this._getJoinFields()];
  }
  async getStyleEditorFields() {
    const sourceFields = await this.getSourceForEditing().getFields();
    return [...sourceFields, ...this._getJoinFields()];
  }
  getIndexPatternIds() {
    const indexPatternIds = this.getSource().getIndexPatternIds();
    this.getValidJoins().forEach(join => {
      indexPatternIds.push(...join.getIndexPatternIds());
    });
    return indexPatternIds;
  }
  getQueryableIndexPatternIds() {
    const indexPatternIds = this.getSource().getQueryableIndexPatternIds();
    this.getValidJoins().forEach(join => {
      indexPatternIds.push(...join.getQueryableIndexPatternIds());
    });
    return indexPatternIds;
  }
  async isFilteredByGlobalTime() {
    if (this.getSource().getApplyGlobalTime() && (await this.getSource().isTimeAware())) {
      return true;
    }
    const joinPromises = this.getValidJoins().map(async join => {
      return join.getRightJoinSource().getApplyGlobalTime() && (await join.getRightJoinSource().isTimeAware());
    });
    return (await Promise.all(joinPromises)).some(isJoinTimeAware => {
      return isJoinTimeAware;
    });
  }
  async _getVectorSourceRequestMeta(isForceRefresh, dataFilters, source, style, isFeatureEditorOpenForLayer) {
    const fieldNames = [...source.getFieldNames(), ...style.getSourceFieldNames(), ...this.getValidJoins().map(join => join.getLeftField().getName())];
    const timesliceMaskFieldName = await source.getTimesliceMaskFieldName();
    if (timesliceMaskFieldName) {
      fieldNames.push(timesliceMaskFieldName);
    }
    return (0, _build_vector_request_meta.buildVectorRequestMeta)(source, fieldNames, dataFilters, this.getQuery(), isForceRefresh, isFeatureEditorOpenForLayer);
  }
  async _syncSourceStyleMeta(syncContext, source, style) {
    const sourceQuery = this.getQuery();
    return this._syncStyleMeta({
      source,
      style,
      sourceQuery: sourceQuery ? sourceQuery : undefined,
      dataRequestId: _constants.SOURCE_META_DATA_REQUEST_ID,
      dynamicStyleProps: style.getDynamicPropertiesArray().filter(dynamicStyleProp => {
        return dynamicStyleProp.getFieldOrigin() === _constants.FIELD_ORIGIN.SOURCE && dynamicStyleProp.isFieldMetaEnabled();
      }),
      ...syncContext
    });
  }
  async _syncStyleMeta({
    source,
    style,
    sourceQuery,
    dataRequestId,
    dynamicStyleProps,
    dataFilters,
    startLoading,
    stopLoading,
    onLoadError,
    registerCancelCallback,
    inspectorAdapters
  }) {
    if (!source.isESSource() || dynamicStyleProps.length === 0) {
      return;
    }
    const dynamicStyleFields = dynamicStyleProps.map(dynamicStyleProp => {
      return `${dynamicStyleProp.getFieldName()}${dynamicStyleProp.getStyleMetaHash()}`;
    });
    const nextMeta = {
      dynamicStyleFields: _lodash.default.uniq(dynamicStyleFields).sort(),
      sourceQuery,
      isTimeAware: this.getCurrentStyle().isTimeAware() && (await source.isTimeAware()),
      timeFilters: dataFilters.timeFilters,
      searchSessionId: dataFilters.searchSessionId
    };
    const prevDataRequest = this.getDataRequest(dataRequestId);
    const canSkipFetch = (0, _can_skip_fetch.canSkipStyleMetaUpdate)({
      prevDataRequest,
      nextMeta
    });
    if (canSkipFetch) {
      return;
    }
    const requestToken = Symbol(`layer-${this.getId()}-${dataRequestId}`);
    try {
      startLoading(dataRequestId, requestToken, nextMeta);
      const layerName = await this.getDisplayName(source);
      const styleMeta = await source.loadStylePropsMeta({
        layerName,
        style,
        dynamicStyleProps,
        registerCancelCallback: registerCancelCallback.bind(null, requestToken),
        sourceQuery: nextMeta.sourceQuery,
        timeFilters: nextMeta.timeFilters,
        searchSessionId: dataFilters.searchSessionId,
        inspectorAdapters,
        executionContext: dataFilters.executionContext
      });
      stopLoading(dataRequestId, requestToken, styleMeta, nextMeta);
    } catch (error) {
      if (!(error instanceof _data_request.DataRequestAbortError)) {
        onLoadError(dataRequestId, requestToken, error.message);
      }
      throw error;
    }
  }
  async _syncSourceFormatters(syncContext, source, style) {
    return this._syncFormatters({
      source,
      dataRequestId: _constants.SOURCE_FORMATTERS_DATA_REQUEST_ID,
      fields: style.getDynamicPropertiesArray().filter(dynamicStyleProp => {
        return dynamicStyleProp.getFieldOrigin() === _constants.FIELD_ORIGIN.SOURCE;
      }).map(dynamicStyleProp => {
        return dynamicStyleProp.getField();
      }),
      ...syncContext
    });
  }
  async _syncFormatters({
    source,
    dataRequestId,
    fields,
    startLoading,
    stopLoading,
    onLoadError
  }) {
    if (fields.length === 0) {
      return;
    }
    const fieldNames = fields.map(field => {
      return field.getName();
    });
    const nextMeta = {
      fieldNames: _lodash.default.uniq(fieldNames).sort()
    };
    const prevDataRequest = this.getDataRequest(dataRequestId);
    const canSkipUpdate = (0, _can_skip_fetch.canSkipFormattersUpdate)({
      prevDataRequest,
      nextMeta
    });
    if (canSkipUpdate) {
      return;
    }
    const requestToken = Symbol(`layer-${this.getId()}-${dataRequestId}`);
    try {
      startLoading(dataRequestId, requestToken, nextMeta);
      const formatters = {};
      const promises = fields.filter(field => {
        return field.canValueBeFormatted();
      }).map(async field => {
        formatters[field.getName()] = await source.createFieldFormatter(field);
      });
      await Promise.all(promises);
      stopLoading(dataRequestId, requestToken, formatters, nextMeta);
    } catch (error) {
      onLoadError(dataRequestId, requestToken, error.message);
      throw error;
    }
  }
  async _syncJoin({
    join,
    startLoading,
    stopLoading,
    onLoadError,
    registerCancelCallback,
    dataFilters,
    isForceRefresh,
    isFeatureEditorOpenForLayer,
    inspectorAdapters
  }) {
    const joinSource = join.getRightJoinSource();
    const sourceDataId = join.getSourceDataRequestId();
    const requestToken = Symbol(`layer-join-refresh:${this.getId()} - ${sourceDataId}`);
    const joinRequestMeta = (0, _build_vector_request_meta.buildVectorRequestMeta)(joinSource, joinSource.getFieldNames(), dataFilters, joinSource.getWhereQuery(), isForceRefresh, isFeatureEditorOpenForLayer);
    const prevDataRequest = this.getDataRequest(sourceDataId);
    const canSkipFetch = await (0, _can_skip_fetch.canSkipSourceUpdate)({
      source: joinSource,
      prevDataRequest,
      nextRequestMeta: joinRequestMeta,
      extentAware: false,
      // join-sources are term-aggs that are spatially unaware (e.g. ESTermSource/TableSource).
      getUpdateDueToTimeslice: () => {
        return true;
      }
    });
    if (canSkipFetch) {
      return {
        dataHasChanged: false,
        join,
        propertiesMap: prevDataRequest === null || prevDataRequest === void 0 ? void 0 : prevDataRequest.getData()
      };
    }
    try {
      startLoading(sourceDataId, requestToken, joinRequestMeta);
      const leftSourceName = await this._source.getDisplayName();
      const propertiesMap = await joinSource.getPropertiesMap(joinRequestMeta, leftSourceName, join.getLeftField().getName(), registerCancelCallback.bind(null, requestToken), inspectorAdapters);
      stopLoading(sourceDataId, requestToken, propertiesMap);
      return {
        dataHasChanged: true,
        join,
        propertiesMap
      };
    } catch (error) {
      if (!(error instanceof _data_request.DataRequestAbortError)) {
        onLoadError(sourceDataId, requestToken, `Join error: ${error.message}`);
      }
      throw error;
    }
  }
  async _syncJoins(syncContext, style) {
    const joinSyncs = this.getValidJoins().map(async join => {
      await this._syncJoinStyleMeta(syncContext, join, style);
      await this._syncJoinFormatters(syncContext, join, style);
      return this._syncJoin({
        join,
        ...syncContext
      });
    });
    return await Promise.all(joinSyncs);
  }
  async _syncJoinStyleMeta(syncContext, join, style) {
    const joinSource = join.getRightJoinSource();
    return this._syncStyleMeta({
      source: joinSource,
      style,
      sourceQuery: joinSource.getWhereQuery(),
      dataRequestId: join.getSourceMetaDataRequestId(),
      dynamicStyleProps: this.getCurrentStyle().getDynamicPropertiesArray().filter(dynamicStyleProp => {
        const matchingField = joinSource.getFieldByName(dynamicStyleProp.getFieldName());
        return dynamicStyleProp.getFieldOrigin() === _constants.FIELD_ORIGIN.JOIN && !!matchingField && dynamicStyleProp.isFieldMetaEnabled();
      }),
      ...syncContext
    });
  }
  async _syncJoinFormatters(syncContext, join, style) {
    const joinSource = join.getRightJoinSource();
    return this._syncFormatters({
      source: joinSource,
      dataRequestId: join.getSourceFormattersDataRequestId(),
      fields: style.getDynamicPropertiesArray().filter(dynamicStyleProp => {
        const matchingField = joinSource.getFieldByName(dynamicStyleProp.getFieldName());
        return dynamicStyleProp.getFieldOrigin() === _constants.FIELD_ORIGIN.JOIN && !!matchingField;
      }).map(dynamicStyleProp => {
        return dynamicStyleProp.getField();
      }),
      ...syncContext
    });
  }
  async _syncSupportsFeatureEditing({
    syncContext,
    source
  }) {
    if (syncContext.dataFilters.isReadOnly) {
      return;
    }
    const {
      startLoading,
      stopLoading,
      onLoadError
    } = syncContext;
    const dataRequestId = SUPPORTS_FEATURE_EDITING_REQUEST_ID;
    const requestToken = Symbol(`layer-${this.getId()}-${dataRequestId}`);
    const prevDataRequest = this.getDataRequest(dataRequestId);
    if (prevDataRequest) {
      return;
    }
    try {
      startLoading(dataRequestId, requestToken);
      const supportsFeatureEditing = await source.supportsFeatureEditing();
      stopLoading(dataRequestId, requestToken, {
        supportsFeatureEditing
      });
    } catch (error) {
      onLoadError(dataRequestId, requestToken, error.message);
      throw error;
    }
  }
  _getJoinFilterExpression() {
    return undefined;
  }
  _createMasks() {
    const masks = [];
    const source = this.getSource();
    if ('getMetricFields' in source) {
      const metricFields = source.getMetricFields();
      metricFields.forEach(metricField => {
        const maskDescriptor = metricField.getMask();
        if (maskDescriptor) {
          masks.push(new _mask.Mask({
            esAggField: metricField,
            isGeometrySourceMvt: source.isMvt(),
            ...maskDescriptor
          }));
        }
      });
    }
    this.getValidJoins().forEach(join => {
      const rightSource = join.getRightJoinSource();
      if ('getMetricFields' in rightSource) {
        const metricFields = rightSource.getMetricFields();
        metricFields.forEach(metricField => {
          const maskDescriptor = metricField.getMask();
          if (maskDescriptor) {
            masks.push(new _mask.Mask({
              esAggField: metricField,
              isGeometrySourceMvt: source.isMvt(),
              ...maskDescriptor
            }));
          }
        });
      }
    });
    return masks;
  }
  getMasks() {
    return this._masks;
  }

  // feature-state is not supported in filter expressions
  // https://github.com/mapbox/mapbox-gl-js/issues/8487
  // therefore, masking must be accomplished via setting opacity paint property (hack)
  _getAlphaExpression() {
    const maskCaseExpressions = [];
    this.getMasks().forEach(mask => {
      // case expressions require 2 parts
      // 1) condition expression
      maskCaseExpressions.push(mask.getMatchMaskedExpression());
      // 2) output. 0 opacity styling "hides" feature
      maskCaseExpressions.push(0);
    });
    return maskCaseExpressions.length ? ['case', ...maskCaseExpressions, this.getAlpha()] : this.getAlpha();
  }
  _setMbPointsProperties(mbMap, mvtSourceLayer, timesliceMaskConfig) {
    const sourceId = this.getId();
    const labelLayerId = this._getMbLabelLayerId();
    const pointLayerId = this._getMbPointLayerId();
    const symbolLayerId = this._getMbSymbolLayerId();
    const pointLayer = mbMap.getLayer(pointLayerId);
    const symbolLayer = mbMap.getLayer(symbolLayerId);

    //
    // Create marker layer
    // "circle" layer type for points
    // "symbol" layer type for icons
    //
    let markerLayerId;
    if (this.getCurrentStyle().arePointsSymbolizedAsCircles()) {
      markerLayerId = pointLayerId;
      if (!pointLayer) {
        const mbLayer = {
          id: pointLayerId,
          type: 'circle',
          source: sourceId,
          paint: {}
        };
        if (mvtSourceLayer) {
          mbLayer['source-layer'] = mvtSourceLayer;
        }
        mbMap.addLayer(mbLayer, labelLayerId);
      }
      if (symbolLayer) {
        mbMap.setLayoutProperty(symbolLayerId, 'visibility', 'none');
      }
    } else {
      markerLayerId = symbolLayerId;
      if (!symbolLayer) {
        const mbLayer = {
          id: symbolLayerId,
          type: 'symbol',
          source: sourceId
        };
        if (mvtSourceLayer) {
          mbLayer['source-layer'] = mvtSourceLayer;
        }
        mbMap.addLayer(mbLayer, labelLayerId);
      }
      if (pointLayer) {
        mbMap.setLayoutProperty(pointLayerId, 'visibility', 'none');
      }
    }
    const isSourceGeoJson = !this.getSource().isMvt();
    const filterExpr = (0, _mb_filter_expressions.getPointFilterExpression)(isSourceGeoJson, this.getSource().isESSource(), this._getJoinFilterExpression(), timesliceMaskConfig);
    if (!_lodash.default.isEqual(filterExpr, mbMap.getFilter(markerLayerId))) {
      mbMap.setFilter(markerLayerId, filterExpr);
    }
    if (this.getCurrentStyle().arePointsSymbolizedAsCircles()) {
      this.getCurrentStyle().setMBPaintPropertiesForPoints({
        alpha: this._getAlphaExpression(),
        mbMap,
        pointLayerId: markerLayerId
      });
    } else {
      this.getCurrentStyle().setMBSymbolPropertiesForPoints({
        alpha: this._getAlphaExpression(),
        mbMap,
        symbolLayerId: markerLayerId
      });
    }
    this.syncVisibilityWithMb(mbMap, markerLayerId);
    mbMap.setLayerZoomRange(markerLayerId, this.getMinZoom(), this.getMaxZoom());
  }
  _setMbLinePolygonProperties(mbMap, mvtSourceLayer, timesliceMaskConfig) {
    const sourceId = this.getId();
    const labelLayerId = this._getMbLabelLayerId();
    const fillLayerId = this._getMbPolygonLayerId();
    const lineLayerId = this._getMbLineLayerId();
    if (!mbMap.getLayer(fillLayerId)) {
      const mbLayer = {
        id: fillLayerId,
        type: 'fill',
        source: sourceId,
        paint: {}
      };
      if (mvtSourceLayer) {
        mbLayer['source-layer'] = mvtSourceLayer;
      }
      mbMap.addLayer(mbLayer, labelLayerId);
    }
    if (!mbMap.getLayer(lineLayerId)) {
      const mbLayer = {
        id: lineLayerId,
        type: 'line',
        source: sourceId,
        paint: {}
      };
      if (mvtSourceLayer) {
        mbLayer['source-layer'] = mvtSourceLayer;
      }
      mbMap.addLayer(mbLayer, labelLayerId);
    }
    this.getCurrentStyle().setMBPaintProperties({
      alpha: this._getAlphaExpression(),
      mbMap,
      fillLayerId,
      lineLayerId
    });
    const joinFilter = this._getJoinFilterExpression();
    this.syncVisibilityWithMb(mbMap, fillLayerId);
    mbMap.setLayerZoomRange(fillLayerId, this.getMinZoom(), this.getMaxZoom());
    const fillFilterExpr = (0, _mb_filter_expressions.getFillFilterExpression)(joinFilter, timesliceMaskConfig);
    if (!_lodash.default.isEqual(fillFilterExpr, mbMap.getFilter(fillLayerId))) {
      mbMap.setFilter(fillLayerId, fillFilterExpr);
    }
    this.syncVisibilityWithMb(mbMap, lineLayerId);
    mbMap.setLayerZoomRange(lineLayerId, this.getMinZoom(), this.getMaxZoom());
    const lineFilterExpr = (0, _mb_filter_expressions.getLineFilterExpression)(joinFilter, timesliceMaskConfig);
    if (!_lodash.default.isEqual(lineFilterExpr, mbMap.getFilter(lineLayerId))) {
      mbMap.setFilter(lineLayerId, lineFilterExpr);
    }
  }
  _setMbLabelProperties(mbMap, mvtSourceLayer, timesliceMaskConfig) {
    const labelLayerId = this._getMbLabelLayerId();
    const labelLayer = mbMap.getLayer(labelLayerId);
    if (!labelLayer) {
      const mbLayer = {
        id: labelLayerId,
        type: 'symbol',
        source: this.getId()
      };
      if (mvtSourceLayer) {
        mbLayer['source-layer'] = mvtSourceLayer;
      }
      mbMap.addLayer(mbLayer);
    }
    const isSourceGeoJson = !this.getSource().isMvt();
    const filterExpr = (0, _mb_filter_expressions.getLabelFilterExpression)(isSourceGeoJson, this.getSource().isESSource(), this._getJoinFilterExpression(), timesliceMaskConfig);
    if (!_lodash.default.isEqual(filterExpr, mbMap.getFilter(labelLayerId))) {
      mbMap.setFilter(labelLayerId, filterExpr);
    }
    this.getCurrentStyle().setMBPropertiesForLabelText({
      alpha: this._getAlphaExpression(),
      mbMap,
      textLayerId: labelLayerId
    });
    this.syncVisibilityWithMb(mbMap, labelLayerId);
  }
  _getMbPointLayerId() {
    return this.makeMbLayerId('circle');
  }
  _getMbLabelLayerId() {
    return this.makeMbLayerId('label');
  }
  _getMbSymbolLayerId() {
    return this.makeMbLayerId('symbol');
  }
  _getMbLineLayerId() {
    return this.makeMbLayerId('line');
  }
  _getMbPolygonLayerId() {
    return this.makeMbLayerId('fill');
  }
  getMbTooltipLayerIds() {
    return [this._getMbPointLayerId(), this._getMbLabelLayerId(), this._getMbSymbolLayerId(), this._getMbLineLayerId(), this._getMbPolygonLayerId()];
  }
  getMbLayerIds() {
    return this.getMbTooltipLayerIds();
  }
  ownsMbLayerId(mbLayerId) {
    return this.getMbLayerIds().includes(mbLayerId);
  }
  ownsMbSourceId(mbSourceId) {
    return this.getId() === mbSourceId;
  }
  _addJoinsToSourceTooltips(tooltipsFromSource) {
    for (let i = 0; i < tooltipsFromSource.length; i++) {
      const tooltipProperty = tooltipsFromSource[i];
      const matchingJoins = [];
      for (let j = 0; j < this.getJoins().length; j++) {
        if (this.getJoins()[j].getLeftField().getName() === tooltipProperty.getPropertyKey()) {
          matchingJoins.push(this.getJoins()[j]);
        }
      }
      if (matchingJoins.length) {
        tooltipsFromSource[i] = new _join_tooltip_property.JoinTooltipProperty(tooltipProperty, matchingJoins);
      }
    }
  }
  async getPropertiesForTooltip(properties, executionContext) {
    const vectorSource = this.getSource();
    let allProperties = await vectorSource.getTooltipProperties(properties, executionContext);
    this._addJoinsToSourceTooltips(allProperties);
    for (let i = 0; i < this.getJoins().length; i++) {
      const propsFromJoin = await this.getJoins()[i].getTooltipProperties(properties, executionContext);
      allProperties = [...allProperties, ...propsFromJoin];
    }
    return allProperties;
  }

  /**
   * Check if there are any properties we can display in a tooltip. If `false` the "Show tooltips" switch
   * is disabled in Layer settings.
   * @returns {boolean}
   */
  canShowTooltip() {
    return this.getSource().hasTooltipProperties() || this.getJoins().length > 0;
  }

  /**
   * Users can toggle tooltips on hover or click in the Layer settings. Tooltips are enabled by default.
   * @returns {boolean}
   */
  areTooltipsDisabled() {
    var _this$_descriptor$dis;
    return (_this$_descriptor$dis = this._descriptor.disableTooltips) !== null && _this$_descriptor$dis !== void 0 ? _this$_descriptor$dis : false;
  }
  getFeatureId(feature) {
    throw new Error('Should implement AbstractVectorLayer#getFeatureId');
  }
  getFeatureById(id) {
    throw new Error('Should implement AbstractVectorLayer#getFeatureById');
  }
  async getLicensedFeatures() {
    return await this._source.getLicensedFeatures();
  }
  async addFeature(geometry) {
    const layerSource = this.getSource();
    const defaultFields = await layerSource.getDefaultFields();
    await layerSource.addFeature(geometry, defaultFields);
  }
  async deleteFeature(featureId) {
    const layerSource = this.getSource();
    await layerSource.deleteFeature(featureId);
  }
  async getStyleMetaDescriptorFromLocalFeatures() {
    throw new Error('Should implement AbstractVectorLayer#getStyleMetaDescriptorFromLocalFeatures');
  }
}
exports.AbstractVectorLayer = AbstractVectorLayer;