"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.syncGeojsonSourceData = syncGeojsonSourceData;
var _constants = require("../../../../../common/constants");
var _data_request = require("../../../util/data_request");
var _get_centroid_features = require("./get_centroid_features");
var _can_skip_fetch = require("../../../util/can_skip_fetch");
var _assign_feature_ids = require("./assign_feature_ids");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function syncGeojsonSourceData({
  layerId,
  layerName,
  prevDataRequest,
  requestMeta,
  syncContext,
  source,
  getUpdateDueToTimeslice
}) {
  const {
    startLoading,
    stopLoading,
    onLoadError,
    registerCancelCallback,
    isRequestStillActive
  } = syncContext;
  const dataRequestId = _constants.SOURCE_DATA_REQUEST_ID;
  const requestToken = Symbol(`${layerId}-${dataRequestId}`);
  const canSkipFetch = syncContext.forceRefreshDueToDrawing ? false : await (0, _can_skip_fetch.canSkipSourceUpdate)({
    source,
    prevDataRequest,
    nextRequestMeta: requestMeta,
    extentAware: source.isFilterByMapBounds(),
    getUpdateDueToTimeslice
  });
  if (canSkipFetch) {
    return {
      refreshed: false,
      featureCollection: prevDataRequest ? prevDataRequest.getData() : _constants.EMPTY_FEATURE_COLLECTION
    };
  }
  try {
    startLoading(dataRequestId, requestToken, requestMeta);
    const {
      data: sourceFeatureCollection,
      meta
    } = await source.getGeoJsonWithMeta(layerName, requestMeta, registerCancelCallback.bind(null, requestToken), () => {
      return isRequestStillActive(dataRequestId, requestToken);
    }, syncContext.inspectorAdapters);
    const layerFeatureCollection = (0, _assign_feature_ids.assignFeatureIds)(sourceFeatureCollection);
    const supportedShapes = await source.getSupportedShapeTypes();
    if (supportedShapes.includes(_constants.VECTOR_SHAPE_TYPE.LINE) || supportedShapes.includes(_constants.VECTOR_SHAPE_TYPE.POLYGON)) {
      layerFeatureCollection.features.push(...(0, _get_centroid_features.getCentroidFeatures)(layerFeatureCollection));
    }
    const responseMeta = meta ? {
      ...meta
    } : {};
    if (requestMeta.applyGlobalTime && (await source.isTimeAware())) {
      const timesliceMaskField = await source.getTimesliceMaskFieldName();
      if (timesliceMaskField) {
        responseMeta.timesliceMaskField = timesliceMaskField;
      }
    }
    stopLoading(dataRequestId, requestToken, layerFeatureCollection, responseMeta);
    return {
      refreshed: true,
      featureCollection: layerFeatureCollection
    };
  } catch (error) {
    if (!(error instanceof _data_request.DataRequestAbortError)) {
      onLoadError(dataRequestId, requestToken, error.message);
    }
    throw error;
  }
}