"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RasterTileLayer = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = _interopRequireDefault(require("lodash"));
var _layer = require("../layer");
var _constants = require("../../../../common/constants");
var _tile_style = require("../../styles/tile/tile_style");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class RasterTileLayer extends _layer.AbstractLayer {
  static createDescriptor(options) {
    const tileLayerDescriptor = super.createDescriptor(options);
    tileLayerDescriptor.type = _constants.LAYER_TYPE.RASTER_TILE;
    tileLayerDescriptor.alpha = _lodash.default.get(options, 'alpha', 1);
    tileLayerDescriptor.style = {
      type: _constants.LAYER_STYLE_TYPE.TILE
    };
    return tileLayerDescriptor;
  }
  constructor({
    source,
    layerDescriptor
  }) {
    super({
      source,
      layerDescriptor
    });
    (0, _defineProperty2.default)(this, "_style", void 0);
    this._style = new _tile_style.TileStyle();
  }
  _isTiled() {
    // Uses tiled maplibre source 'raster'
    return true;
  }
  getSource() {
    return super.getSource();
  }
  async hasLegendDetails() {
    const source = this.getSource();
    return await source.hasLegendDetails();
  }
  renderLegendDetails() {
    const dataRequest = this.getSourceDataRequest();
    const source = this.getSource();
    return source.renderLegendDetails(dataRequest);
  }
  getStyleForEditing() {
    return this._style;
  }
  getStyle() {
    return this._style;
  }
  getCurrentStyle() {
    return this._style;
  }
  async syncData({
    startLoading,
    stopLoading,
    onLoadError,
    dataFilters,
    isForceRefresh
  }) {
    const source = this.getSource();
    const nextMeta = {
      ...dataFilters,
      applyGlobalQuery: source.getApplyGlobalQuery(),
      applyGlobalTime: source.getApplyGlobalTime(),
      applyForceRefresh: source.isESSource() ? source.getApplyForceRefresh() : false,
      sourceQuery: this.getQuery() || undefined,
      isForceRefresh
    };
    const prevDataRequest = this.getSourceDataRequest();
    if (prevDataRequest) {
      const canSkip = await source.canSkipSourceUpdate(prevDataRequest, nextMeta);
      if (canSkip) return;
    }
    const requestToken = Symbol(`layer-source-refresh:${this.getId()} - source`);
    try {
      startLoading(_constants.SOURCE_DATA_REQUEST_ID, requestToken, nextMeta);
      const data = {
        url: await source.getUrlTemplate(nextMeta)
      };
      stopLoading(_constants.SOURCE_DATA_REQUEST_ID, requestToken, data, {});
    } catch (error) {
      onLoadError(_constants.SOURCE_DATA_REQUEST_ID, requestToken, error.message);
    }
  }
  _getMbLayerId() {
    return this.makeMbLayerId('raster');
  }
  getMbLayerIds() {
    return [this._getMbLayerId()];
  }
  ownsMbLayerId(mbLayerId) {
    return this._getMbLayerId() === mbLayerId;
  }
  ownsMbSourceId(mbSourceId) {
    return this.getId() === mbSourceId;
  }
  _requiresPrevSourceCleanup(mbMap) {
    const source = this.getSource();
    const mbSource = mbMap.getSource(this.getMbSourceId());
    if (!mbSource) {
      return false;
    }
    const sourceDataRequest = this.getSourceDataRequest();
    if (sourceDataRequest) {
      const data = sourceDataRequest.getData();
      if (data) {
        return source.isSourceStale(mbSource, data);
      }
    }
    return false;
  }
  syncLayerWithMB(mbMap) {
    this._removeStaleMbSourcesAndLayers(mbMap);
    const source = mbMap.getSource(this.getId());
    const mbLayerId = this._getMbLayerId();
    if (!source) {
      const sourceDataRequest = this.getSourceDataRequest();
      if (!sourceDataRequest) {
        // this is possible if the layer was invisible at startup.
        // the actions will not perform any data=syncing as an optimization when a layer is invisible
        // when turning the layer back into visible, it's possible the url has not been resovled yet.
        return;
      }
      const tmsSourceData = sourceDataRequest.getData();
      if (!tmsSourceData || !tmsSourceData.url) {
        return;
      }
      const mbSourceId = this.getMbSourceId();
      mbMap.addSource(mbSourceId, {
        type: 'raster',
        tiles: [tmsSourceData.url],
        tileSize: 256,
        scheme: 'xyz'
      });
      mbMap.addLayer({
        id: mbLayerId,
        type: 'raster',
        source: mbSourceId,
        minzoom: this._descriptor.minZoom,
        maxzoom: this._descriptor.maxZoom
      });
    }
    this._setTileLayerProperties(mbMap, mbLayerId);
  }
  _setTileLayerProperties(mbMap, mbLayerId) {
    this.syncVisibilityWithMb(mbMap, mbLayerId);
    mbMap.setLayerZoomRange(mbLayerId, this.getMinZoom(), this.getMaxZoom());
    mbMap.setPaintProperty(mbLayerId, 'raster-opacity', this.getAlpha());
  }
  getLayerTypeIconName() {
    return 'grid';
  }
  isBasemap(order) {
    return order === 0;
  }
  async isFilteredByGlobalTime() {
    return this.getSource().getApplyGlobalTime() && (await this.getSource().isTimeAware());
  }
}
exports.RasterTileLayer = RasterTileLayer;