"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AbstractLayer = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = _interopRequireDefault(require("lodash"));
var _react = _interopRequireDefault(require("react"));
var _eui = require("@elastic/eui");
var _uuid = require("uuid");
var _data_request = require("../util/data_request");
var _constants = require("../../../common/constants");
var _copy_persistent_state = require("../../reducers/copy_persistent_state");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/* eslint-disable @typescript-eslint/consistent-type-definitions */

class AbstractLayer {
  static createDescriptor(options) {
    return {
      ...options,
      sourceDescriptor: options.sourceDescriptor ? options.sourceDescriptor : null,
      __dataRequests: _lodash.default.get(options, '__dataRequests', []),
      id: _lodash.default.get(options, 'id', (0, _uuid.v4)()),
      label: options.label && options.label.length > 0 ? options.label : null,
      minZoom: _lodash.default.get(options, 'minZoom', _constants.MIN_ZOOM),
      maxZoom: _lodash.default.get(options, 'maxZoom', _constants.MAX_ZOOM),
      alpha: _lodash.default.get(options, 'alpha', 0.75),
      visible: _lodash.default.get(options, 'visible', true),
      style: _lodash.default.get(options, 'style', null),
      includeInFitToBounds: typeof options.includeInFitToBounds === 'boolean' ? options.includeInFitToBounds : true
    };
  }
  constructor({
    layerDescriptor,
    source
  }) {
    (0, _defineProperty2.default)(this, "_descriptor", void 0);
    (0, _defineProperty2.default)(this, "_source", void 0);
    (0, _defineProperty2.default)(this, "_dataRequests", void 0);
    this._descriptor = AbstractLayer.createDescriptor(layerDescriptor);
    this._source = source;
    if (this._descriptor.__dataRequests) {
      this._dataRequests = this._descriptor.__dataRequests.map(dataRequest => new _data_request.DataRequest(dataRequest));
    } else {
      this._dataRequests = [];
    }
  }
  static getBoundDataForSource(mbMap, sourceId) {
    // @ts-expect-error
    const mbStyle = mbMap.getStyle();
    return mbStyle.sources[sourceId].data;
  }
  getDescriptor() {
    return this._descriptor;
  }
  async cloneDescriptor() {
    const clonedDescriptor = (0, _copy_persistent_state.copyPersistentState)(this._descriptor);
    // layer id is uuid used to track styles/layers in mapbox
    clonedDescriptor.id = (0, _uuid.v4)();
    const displayName = await this.getDisplayName();
    clonedDescriptor.label = `Clone of ${displayName}`;
    clonedDescriptor.sourceDescriptor = this.getSource().cloneDescriptor();
    return [clonedDescriptor];
  }
  makeMbLayerId(layerNameSuffix) {
    return `${this.getId()}${_constants.MB_SOURCE_ID_LAYER_ID_PREFIX_DELIMITER}${layerNameSuffix}`;
  }
  isPreviewLayer() {
    return !!this._descriptor.__isPreviewLayer;
  }
  supportsElasticsearchFilters() {
    return this.getSource().isESSource();
  }
  async supportsFitToBounds() {
    return await this.getSource().supportsFitToBounds();
  }
  async isFittable() {
    return (await this.supportsFitToBounds()) && this.isVisible() && this.isIncludeInFitToBounds();
  }
  isIncludeInFitToBounds() {
    return !!this._descriptor.includeInFitToBounds;
  }
  async isFilteredByGlobalTime() {
    return false;
  }
  async getDisplayName(source) {
    if (this._descriptor.label) {
      return this._descriptor.label;
    }
    const sourceDisplayName = source ? await source.getDisplayName() : await this.getSource().getDisplayName();
    return sourceDisplayName || `Layer ${this._descriptor.id}`;
  }
  async getAttributions() {
    if (this.hasErrors() || !this.isVisible()) {
      return [];
    }
    const attributionProvider = this.getSource().getAttributionProvider();
    if (attributionProvider) {
      return attributionProvider();
    }
    return this._descriptor.attribution !== undefined ? [this._descriptor.attribution] : [];
  }
  getStyleForEditing() {
    throw new Error('Should implement AbstractLayer#getStyleForEditing');
  }
  getStyle() {
    throw new Error('Should implement AbstractLayer#getStyle');
  }
  getCurrentStyle() {
    throw new Error('Should implement AbstractLayer#getCurrentStyle');
  }
  getLabel() {
    return this._descriptor.label ? this._descriptor.label : '';
  }
  getLocale() {
    return null;
  }
  getLayerIcon(isTocIcon) {
    return {
      icon: /*#__PURE__*/_react.default.createElement(_eui.EuiIcon, {
        size: "m",
        type: this.getLayerTypeIconName()
      })
    };
  }
  async hasLegendDetails() {
    return false;
  }
  renderLegendDetails() {
    return null;
  }
  getId() {
    return this._descriptor.id;
  }
  getSource() {
    return this._source;
  }
  getSourceForEditing() {
    return this._source;
  }
  isVisible() {
    return !!this._descriptor.visible;
  }
  showAtZoomLevel(zoom) {
    return zoom >= this.getMinZoom() && zoom <= this.getMaxZoom();
  }
  getMinZoom() {
    return typeof this._descriptor.minZoom === 'number' ? this._descriptor.minZoom : _constants.MIN_ZOOM;
  }
  getMaxZoom() {
    return typeof this._descriptor.maxZoom === 'number' ? this._descriptor.maxZoom : _constants.MAX_ZOOM;
  }
  getMinSourceZoom() {
    return this._source.getMinZoom();
  }
  getMbSourceId() {
    return this.getId();
  }
  _requiresPrevSourceCleanup(mbMap) {
    return false;
  }
  _removeStaleMbSourcesAndLayers(mbMap) {
    if (this._requiresPrevSourceCleanup(mbMap)) {
      // @ts-expect-error
      const mbStyle = mbMap.getStyle();
      // @ts-expect-error
      mbStyle.layers.forEach(mbLayer => {
        if (this.ownsMbLayerId(mbLayer.id)) {
          // @ts-expect-error
          mbMap.removeLayer(mbLayer.id);
        }
      });
      Object.keys(mbStyle.sources).some(mbSourceId => {
        if (this.ownsMbSourceId(mbSourceId)) {
          // @ts-expect-error
          mbMap.removeSource(mbSourceId);
        }
      });
    }
  }
  getAlpha() {
    return typeof this._descriptor.alpha === 'number' ? this._descriptor.alpha : 1;
  }
  getQuery() {
    return this._descriptor.query ? this._descriptor.query : null;
  }
  async getImmutableSourceProperties() {
    const source = this.getSource();
    return await source.getImmutableProperties();
  }
  renderSourceSettingsEditor(sourceEditorArgs) {
    return this.getSourceForEditing().renderSourceSettingsEditor(sourceEditorArgs);
  }
  getPrevRequestToken(dataId) {
    const prevDataRequest = this.getDataRequest(dataId);
    if (!prevDataRequest) {
      return;
    }
    return prevDataRequest.getRequestToken();
  }
  getInFlightRequestTokens() {
    if (!this._dataRequests) {
      return [];
    }
    const requestTokens = this._dataRequests.map(dataRequest => dataRequest.getRequestToken());

    // Compact removes all the undefineds
    return _lodash.default.compact(requestTokens);
  }
  getSourceDataRequest() {
    return this.getDataRequest(_constants.SOURCE_DATA_REQUEST_ID);
  }
  getDataRequest(id) {
    return this._dataRequests.find(dataRequest => dataRequest.getDataId() === id);
  }
  isLayerLoading(zoom) {
    if (!this.isVisible() || !this.showAtZoomLevel(zoom)) {
      return false;
    }
    const hasOpenDataRequests = this._dataRequests.some(dataRequest => dataRequest.isLoading());
    if (this._isTiled()) {
      return hasOpenDataRequests || this._descriptor.__areTilesLoaded === undefined || !this._descriptor.__areTilesLoaded;
    }
    return !this.getSourceDataRequest() ? true // layer is loading until source data request has been created
    : hasOpenDataRequests;
  }
  hasErrors() {
    return _lodash.default.get(this._descriptor, '__isInErrorState', false);
  }
  getErrors() {
    return this.hasErrors() && this._descriptor.__errorMessage ? this._descriptor.__errorMessage : '';
  }
  async syncData(syncContext) {
    // no-op by default
  }
  getMbLayerIds() {
    throw new Error('Should implement AbstractLayer#getMbLayerIds');
  }
  ownsMbLayerId(layerId) {
    throw new Error('Should implement AbstractLayer#ownsMbLayerId');
  }
  ownsMbSourceId(mbSourceId) {
    throw new Error('Should implement AbstractLayer#ownsMbSourceId');
  }
  syncLayerWithMB(mbMap) {
    throw new Error('Should implement AbstractLayer#syncLayerWithMB');
  }
  getLayerTypeIconName() {
    throw new Error('should implement Layer#getLayerTypeIconName');
  }
  async getBounds(getDataRequestContext) {
    return null;
  }
  renderStyleEditor(onStyleDescriptorChange, onCustomIconsChange) {
    const style = this.getStyleForEditing();
    if (!style) {
      return null;
    }
    return style.renderEditor(onStyleDescriptorChange, onCustomIconsChange);
  }
  getIndexPatternIds() {
    return [];
  }
  getQueryableIndexPatternIds() {
    return [];
  }
  syncVisibilityWithMb(mbMap, mbLayerId) {
    // @ts-expect-error
    mbMap.setLayoutProperty(mbLayerId, 'visibility', this.isVisible() ? 'visible' : 'none');
  }
  getType() {
    return this._descriptor.type;
  }
  areLabelsOnTop() {
    return false;
  }
  supportsLabelsOnTop() {
    return false;
  }
  supportsLabelLocales() {
    return false;
  }
  async getLicensedFeatures() {
    return [];
  }
  getGeoFieldNames() {
    const source = this.getSource();
    return source.isESSource() ? [source.getGeoFieldName()] : [];
  }
  async getStyleMetaDescriptorFromLocalFeatures() {
    return null;
  }
  isBasemap(order) {
    return false;
  }
  getParent() {
    return this._descriptor.parent;
  }
  _getMetaFromTiles() {
    return this._descriptor.__metaFromTiles || [];
  }
  _isTiled() {
    throw new Error('Must implement AbstractLayer#_isTiled');
  }
}
exports.AbstractLayer = AbstractLayer;